/**
 * AbstractGlobalWekaClassifierEvaluator.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer;

import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.Token;
import adams.flow.source.GlobalSource;

/**
 * Ancestor for classifier evaluators that make use of a global classifier.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3616 $
 */
public abstract class AbstractGlobalWekaClassifierEvaluator
  extends AbstractWekaClassifierEvaluator {

  /** for serialization. */
  private static final long serialVersionUID = 3440872619963043115L;

  /** the name of the global weka classifier. */
  protected GlobalActorReference m_Classifier;

  /** whether to discard predictions. */
  protected boolean m_DiscardPredictions;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "classifier", "classifier",
	    new GlobalActorReference("WekaClassifier"));

    m_OptionManager.add(
	    "no-predictions", "discardPredictions",
	    false);
  }

  /**
   * Sets the name of the global classifier to use.
   *
   * @param value	the name
   */
  public void setClassifier(GlobalActorReference value) {
    m_Classifier = value;
    reset();
  }

  /**
   * Returns the name of the global classifier in use.
   *
   * @return		the name
   */
  public GlobalActorReference getClassifier() {
    return m_Classifier;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public abstract String classifierTipText();

  /**
   * Sets whether to discard the predictions instead of collecting them
   * for future use, in order to conserve memory.
   *
   * @param value	true if to discard predictions
   */
  public void setDiscardPredictions(boolean value) {
    m_DiscardPredictions = value;
    reset();
  }

  /**
   * Returns whether to discard the predictions in order to preserve memory.
   *
   * @return		true if predictions discarded
   */
  public boolean getDiscardPredictions() {
    return m_DiscardPredictions;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String discardPredictionsTipText() {
    return
        "If enabled, the collection of predictions during evaluation is "
      + "suppressed, wich will conserve memory.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("classifier");

    if (variable != null)
      return variable;
    else if (m_Classifier != null)
      return m_Classifier.toString();
    else
      return null;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;
    String	variable;

    result = super.setUp();

    if (result == null) {
      variable = getOptionManager().getVariableForProperty("classifier");
      if (variable == null) {
	if (m_Classifier.isEmpty())
	  result = "No classifier specified!";
      }
    }

    return result;
  }

  /**
   * Returns an instance of the global classifier.
   *
   * @return		the classifier
   */
  protected weka.classifiers.Classifier getClassifierInstance() {
    GlobalSource		global;
    weka.classifiers.Classifier	result;
    AbstractActor		actor;
    Token			token;

    result = null;

    global = new GlobalSource();
    global.setGlobalName(m_Classifier);
    global.setParent(getParent());
    if (global.setUp() == null) {
      global.execute();
      token = global.output();
      if (token != null) {
	result = (weka.classifiers.Classifier) token.getPayload();
	actor  = global.getGlobalActor();
	global.wrapUp();
	global.cleanUp();
	// fix wrapUp/cleanUp of global actor, e.g., re-registering variable listeners
	if (actor != null)
	  actor.setUp();
      }
    }

    return result;
  }
}
