/*
 * WekaClassSelector.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Vector;

import weka.core.Attribute;
import adams.core.Index;
import adams.core.base.BaseRegExp;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.provenance.ProvenanceSupporter;

/**
 <!-- globalinfo-start -->
 * Sets the class index. Can either honour an already existing one or override it. Also, one can apply the index on a subset of attributes defined by a regular expression applied to the attribute names (one can set the class index to the last attribute that starts with 'att_').
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instances<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.instance.Instance<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instances<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.instance.Instance<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ClassSelector
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-class &lt;java.lang.String&gt; (property: classIndex)
 * &nbsp;&nbsp;&nbsp;The index within the attribute subset defined by the regular expression
 * &nbsp;&nbsp;&nbsp;('first' and 'last' are accepted as well).
 * &nbsp;&nbsp;&nbsp;default: last
 * </pre>
 *
 * <pre>-override (property: override)
 * &nbsp;&nbsp;&nbsp;If set to true, then any existing class index will be overriden; otherwise
 * &nbsp;&nbsp;&nbsp;the class index will only be set if not already set.
 * </pre>
 *
 * <pre>-regex &lt;java.lang.String&gt; (property: regexName)
 * &nbsp;&nbsp;&nbsp;The regular expression used for selecting the subset of attributes.
 * &nbsp;&nbsp;&nbsp;default: .*
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3355 $
 */
public class WekaClassSelector
  extends AbstractWekaInstanceAndWekaInstancesTransformer
  implements ProvenanceSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -3019442578354930841L;

  /** the class index. */
  protected Index m_ClassIndex;

  /** whether to override any set class attribute.  */
  protected boolean m_Override;

  /** the regular expression on the attribute for selecting the sub-set of attributes. */
  protected BaseRegExp m_RegexName;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Sets the class index. Can either honour an already existing one or "
      + "override it. Also, one can apply the index on a subset of attributes "
      + "defined by a regular expression applied to the attribute names (one "
      + "can set the class index to the last attribute that starts with 'att_').";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_ClassIndex = new Index();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "class", "classIndex",
	    "last");

    m_OptionManager.add(
	    "override", "override",
	    false);

    m_OptionManager.add(
	    "regex", "regexName",
	    new BaseRegExp(BaseRegExp.MATCH_ALL));
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = null;

    variable = getOptionManager().getVariableForProperty("classIndex");
    if ((getClassIndex() != null) || (variable != null)) {
      // index
      if (variable != null)
	result = variable;
      else
	result = getClassIndex();

      // override?
      if (getOverride())
	result += "/override";

      // regexName
      variable = getOptionManager().getVariableForProperty("regexName");
      if (variable != null)
	result += "/" + variable;
      else if (!m_RegexName.isEmpty() && !m_RegexName.isMatchAll())
	result += "/" + getRegexName();
    }

    return result;
  }

  /**
   * Sets the class index.
   *
   * @param value	the index
   */
  public void setClassIndex(String value) {
    m_ClassIndex.setIndex(value);
    reset();
  }

  /**
   * Returns the class index.
   *
   * @return		the index
   */
  public String getClassIndex() {
    return m_ClassIndex.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String classIndexTipText() {
    return
        "The index within the attribute subset defined by the regular "
      + "expression ('first' and 'last' are accepted as well).";
  }

  /**
   * Sets whether to override any existing class index or nor.
   *
   * @param value	if true then any class index will be override
   */
  public void setOverride(boolean value) {
    m_Override = value;
    reset();
  }

  /**
   * Returns whether any existing class index will be overriden or not.
   *
   * @return		true if any class index will be overriden
   */
  public boolean getOverride() {
    return m_Override;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String overrideTipText() {
    return
        "If set to true, then any existing class index will be overriden; "
      + "otherwise the class index will only be set if not already set.";
  }

  /**
   * Sets the regular expression for selecting the attributes.
   *
   * @param value	the regex
   */
  public void setRegexName(BaseRegExp value) {
    m_RegexName = value;
    reset();
  }

  /**
   * Returns the regular expression for selecting the attributes.
   *
   * @return		the regex
   */
  public BaseRegExp getRegexName() {
    return m_RegexName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String regexNameTipText() {
    return "The regular expression used for selecting the subset of attributes.";
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    Object			o;
    weka.core.Instances		inst;
    boolean			isInstances;
    Vector<Attribute>		atts;
    int				i;

    result = null;

    // dataset
    o           = m_InputToken.getPayload();
    isInstances = false;
    inst        = null;
    if (o instanceof weka.core.Instances) {
      inst        = (weka.core.Instances) o;
      inst        = new weka.core.Instances(inst);
      isInstances = true;
    }
    else if (o instanceof adams.data.instance.Instance) {
      inst = ((adams.data.instance.Instance) o).getDatasetHeader();
    }
    else if (o instanceof weka.core.Instance) {
      inst = ((weka.core.Instance) o).dataset();
    }
    else {
      result = "Cannot handle object of type " + o.getClass().getName() + "!";
    }

    if (result == null) {
      // determine the attributes that fit the regular expression
      atts = new Vector<Attribute>();
      for (i = 0; i < inst.numAttributes(); i++) {
	if (m_RegexName.isEmpty() || m_RegexName.isMatch(inst.attribute(i).name()))
	  atts.add(inst.attribute(i));
      }

      // class index
      m_ClassIndex.setMax(atts.size());
      if (m_Override || (inst.classIndex() == -1))
	inst.setClassIndex(atts.get(m_ClassIndex.getIntIndex()).index());

      // output
      if (isInstances)
	m_OutputToken = new Token(inst);
      else
	m_OutputToken = new Token(o);

      updateProvenance(m_OutputToken);
    }

    return result;
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled()) {
      cont.setProvenance(m_InputToken.getProvenance());
      cont.addProvenance(new ProvenanceInformation(ActorType.PREPROCESSOR, m_InputToken.getPayload().getClass(), this, m_OutputToken.getPayload().getClass()));
    }
  }
}
