/*
 * WekaEvaluationValuePicker.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import weka.classifiers.Evaluation;
import adams.flow.core.EvaluationStatistic;
import adams.flow.core.Token;


/**
 <!-- globalinfo-start -->
 * Picks a specific value from an evaluation object.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.classifiers.Evaluation<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: EvaluationValuePicker
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-statistic &lt;ELAPSED_TIME_TRAINING|ELAPSED_TIME_TESTING|USERCPU_TIME_TRAINING|USERCPU_TIME_TESTING|SERIALIZED_MODEL_SIZE|SERIALIZED_TRAIN_SET_SIZE|SERIALIZED_TEST_SET_SIZE|NUMBER_OF_TRAINING_INSTANCES|NUMBER_OF_TESTING_INSTANCES|NUMBER_CORRECT|NUMBER_INCORRECT|NUMBER_UNCLASSIFIED|PERCENT_CORRECT|PERCENT_INCORRECT|PERCENT_UNCLASSIFIED|KAPPA_STATISTIC|MEAN_ABSOLUTE_ERROR|ROOT_MEAN_SQUARED_ERROR|RELATIVE_ABSOLUTE_ERROR|ROOT_RELATIVE_SQUARED_ERROR|CORRELATION_COEFFICIENT|SF_PRIOR_ENTROPY|SF_SCHEME_ENTROPY|SF_ENTROPY_GAIN|SF_MEAN_PRIOR_ENTROPY|SF_MEAN_SCHEME_ENTROPY|SF_MEAN_ENTROPY_GAIN|KB_INFORMATION|KB_MEAN_INFORMATION|KB_RELATIVE_INFORMATION|TRUE_POSITIVE_RATE|NUM_TRUE_POSITIVES|FALSE_POSITIVE_RATE|NUM_FALSE_POSITIVES|TRUE_NEGATIVE_RATE|NUM_TRUE_NEGATIVES|FALSE_NEGATIVE_RATE|NUM_FALSE_NEGATIVES|IR_PRECISION|IR_RECALL|F_MEASURE|AREA_UNDER_ROC&gt; (property: statisticValue)
 * &nbsp;&nbsp;&nbsp;The evaluation value to extract.
 * &nbsp;&nbsp;&nbsp;default: PERCENT_CORRECT
 * </pre>
 *
 * <pre>-index &lt;int&gt; (property: classIndex)
 * &nbsp;&nbsp;&nbsp;The class label index (eg used for AUC).
 * &nbsp;&nbsp;&nbsp;default: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2823 $
 */
public class WekaEvaluationValuePicker
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -3113058781746945626L;

  /** the comparison field. */
  protected EvaluationStatistic m_StatisticValue;

  /** the index of the class label. */
  protected int m_ClassIndex;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Picks a specific value from an evaluation object.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "statistic", "statisticValue",
	    EvaluationStatistic.PERCENT_CORRECT);

    m_OptionManager.add(
	    "index", "classIndex",
	    1);
  }

  /**
   * Sets the value to extract.
   *
   * @param value	the value
   */
  public void setStatisticValue(EvaluationStatistic value) {
    m_StatisticValue = value;
    reset();
  }

  /**
   * Returns the value to extract.
   *
   * @return		the value
   */
  public EvaluationStatistic getStatisticValue() {
    return m_StatisticValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String statisticValueTipText() {
    return "The evaluation value to extract.";
  }

  /**
   * Sets the class label index (1-based).
   *
   * @param value	the label index
   */
  public void setClassIndex(int value) {
    m_ClassIndex = value;
    reset();
  }

  /**
   * Returns the current class label index (1-based).
   *
   * @return		the label index
   */
  public int getClassIndex() {
    return m_ClassIndex;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String classIndexTipText() {
    return "The class label index (eg used for AUC).";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("classIndex");
    if (variable != null)
      result = variable;
    else
      result = "" + m_ClassIndex;

    result += ": ";

    variable = getOptionManager().getVariableForProperty("statisticValue");
    if (variable != null)
      result += variable;
    else
      result += m_StatisticValue;

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.classifiers.Evaluation.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Evaluation.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    double	value;
    Evaluation	eval;

    result = null;

    value = Double.NaN;
    eval  = (Evaluation) m_InputToken.getPayload();
    try {
      if (m_StatisticValue == EvaluationStatistic.NUMBER_CORRECT)
	value = eval.correct();
      else if (m_StatisticValue == EvaluationStatistic.NUMBER_INCORRECT)
	value = eval.incorrect();
      else if (m_StatisticValue == EvaluationStatistic.NUMBER_UNCLASSIFIED)
	value = eval.unclassified();
      else if (m_StatisticValue == EvaluationStatistic.PERCENT_CORRECT)
	value = eval.pctCorrect();
      else if (m_StatisticValue == EvaluationStatistic.PERCENT_INCORRECT)
	value = eval.pctIncorrect();
      else if (m_StatisticValue == EvaluationStatistic.PERCENT_UNCLASSIFIED)
	value = eval.pctUnclassified();
      else if (m_StatisticValue == EvaluationStatistic.KAPPA_STATISTIC)
	value = eval.kappa();
      else if (m_StatisticValue == EvaluationStatistic.MEAN_ABSOLUTE_ERROR)
	value = eval.meanAbsoluteError();
      else if (m_StatisticValue == EvaluationStatistic.ROOT_MEAN_SQUARED_ERROR)
	value = eval.rootMeanSquaredError();
      else if (m_StatisticValue == EvaluationStatistic.RELATIVE_ABSOLUTE_ERROR)
	value = eval.relativeAbsoluteError();
      else if (m_StatisticValue == EvaluationStatistic.ROOT_RELATIVE_SQUARED_ERROR)
	value = eval.rootRelativeSquaredError();
      else if (m_StatisticValue == EvaluationStatistic.CORRELATION_COEFFICIENT)
	value = eval.correlationCoefficient();
      else if (m_StatisticValue == EvaluationStatistic.SF_PRIOR_ENTROPY)
	value = eval.SFPriorEntropy();
      else if (m_StatisticValue == EvaluationStatistic.SF_SCHEME_ENTROPY)
	value = eval.SFSchemeEntropy();
      else if (m_StatisticValue == EvaluationStatistic.SF_ENTROPY_GAIN)
	value = eval.SFEntropyGain();
      else if (m_StatisticValue == EvaluationStatistic.SF_MEAN_PRIOR_ENTROPY)
	value = eval.SFMeanPriorEntropy();
      else if (m_StatisticValue == EvaluationStatistic.SF_MEAN_SCHEME_ENTROPY)
	value = eval.SFMeanSchemeEntropy();
      else if (m_StatisticValue == EvaluationStatistic.SF_MEAN_ENTROPY_GAIN)
	value = eval.SFMeanEntropyGain();
      else if (m_StatisticValue == EvaluationStatistic.KB_INFORMATION)
	value = eval.KBInformation();
      else if (m_StatisticValue == EvaluationStatistic.KB_MEAN_INFORMATION)
	value = eval.KBMeanInformation();
      else if (m_StatisticValue == EvaluationStatistic.KB_RELATIVE_INFORMATION)
	value = eval.KBRelativeInformation();
      else if (m_StatisticValue == EvaluationStatistic.TRUE_POSITIVE_RATE)
	value = eval.truePositiveRate(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.NUM_TRUE_POSITIVES)
	value = eval.numTruePositives(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.FALSE_POSITIVE_RATE)
	value = eval.falsePositiveRate(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.NUM_FALSE_POSITIVES)
	value = eval.numFalsePositives(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.TRUE_NEGATIVE_RATE)
	value = eval.trueNegativeRate(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.NUM_TRUE_NEGATIVES)
	value = eval.numTrueNegatives(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.FALSE_NEGATIVE_RATE)
	value = eval.falseNegativeRate(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.NUM_FALSE_NEGATIVES)
	value = eval.numFalseNegatives(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.IR_PRECISION)
	value = eval.precision(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.IR_RECALL)
	value = eval.recall(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.F_MEASURE)
	value = eval.fMeasure(m_ClassIndex);
      else if (m_StatisticValue == EvaluationStatistic.AREA_UNDER_ROC)
	value = eval.areaUnderROC(m_ClassIndex);

      m_OutputToken = new Token(value);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = "Error retrieving value for '" + m_StatisticValue + "':\n" + e;
    }

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Double.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Double.class};
  }
}
