/*
 * WekaExperiment.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.io.File;
import java.util.Vector;

import adams.core.io.PlaceholderFile;
import adams.data.WekaExperimentFile;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Represents a Weka experiment, stored in a file. Can be setup from inside the flow.<br/>
 * Via the input port, additional datasets can be inserted into the experiment. This allows one to define a template experiment and then just run it over various datasets without every having to change the experiment setup itself.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * &nbsp;&nbsp;&nbsp;java.io.File<br/>
 * &nbsp;&nbsp;&nbsp;java.io.File[]<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String[]<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;weka.experiment.Experiment<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Experiment
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-file &lt;adams.data.ExperimentFile&gt; (property: experimentFile)
 * &nbsp;&nbsp;&nbsp;The file the experiment is stored in.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-no-check (property: noCheck)
 * &nbsp;&nbsp;&nbsp;If set to true, the experiment file must not be present at setUp-time; necessary
 * &nbsp;&nbsp;&nbsp;if file gets generated on-the-fly.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2823 $
 */
public class WekaExperiment
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -1127504846960163422L;

  /** the experiment file. */
  protected WekaExperimentFile m_ExperimentFile;

  /** whether not to check for experiment file to exist (e.g., when it generated on the fly). */
  protected boolean m_NoCheck;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Represents a Weka experiment, stored in a file. Can be setup from "
      + "inside the flow.\n"
      + "Via the input port, additional datasets can be inserted into the "
      + "experiment. This allows one to define a template experiment and "
      + "then just run it over various datasets without every having to "
      + "change the experiment setup itself.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "file", "experimentFile",
	    new WekaExperimentFile("."));

    m_OptionManager.add(
	    "no-check", "noCheck",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("experimentFile");

    if (variable != null)
      return variable;
    else if (m_ExperimentFile != null)
      return m_ExperimentFile.toString();
    else
      return null;
  }

  /**
   * Sets the file the experiment is stored in.
   *
   * @param value	the file
   */
  public void setExperimentFile(WekaExperimentFile value) {
    m_ExperimentFile = value;
    reset();
  }

  /**
   * Returns the file the experiment is stored in.
   *
   * @return		the file
   */
  public WekaExperimentFile getExperimentFile() {
    return m_ExperimentFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String experimentFileTipText() {
    return "The file the experiment is stored in.";
  }

  /**
   * Sets whether to avoid the check at setUp time whether the experiment file
   * is present or not.
   *
   * @param value	true if the file need not be there
   */
  public void setNoCheck(boolean value) {
    m_NoCheck = value;
    reset();
  }

  /**
   * Returns whether to avoid the check at setUp time whether the experiment file
   * is present or not.
   *
   * @return		true if the file need not be there
   */
  public boolean getNoCheck() {
    return m_NoCheck;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String noCheckTipText() {
    return
        "If set to true, the experiment file must not be present at setUp-time; "
      + "necessary if file gets generated on-the-fly.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class, java.io.File.class, java.io.File[].class, java.lang.String.class, java.lang.String[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class, File.class, File[].class, String.class, String[].class};
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if (!m_NoCheck) {
	if (!m_ExperimentFile.exists())
	  result = "Experiment file '" + m_ExperimentFile.getAbsolutePath() + "' does not exist!";
      }
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    weka.experiment.Experiment	exp;
    File[]			tmpFiles;
    Vector<File>		files;
    String[]			tmpStr;
    int				i;

    result = null;

    // load experiment
    try {
      exp = weka.experiment.Experiment.read(m_ExperimentFile.getAbsolutePath());
    }
    catch (Exception e) {
      exp    = null;
      result = e.toString();
      getSystemErr().printStackTrace(e);
    }

    // more datasets to add?
    if (m_InputToken != null) {
      // get files
      files = new Vector<File>();
      if (m_InputToken.getPayload() instanceof File) {
	files.add(((File) m_InputToken.getPayload()).getAbsoluteFile());
      }
      else if (m_InputToken.getPayload() instanceof File[]) {
	tmpFiles = (File[]) m_InputToken.getPayload();
	for (i = 0; i < tmpFiles.length; i++)
	  files.add(tmpFiles[i].getAbsoluteFile());
      }
      else if (m_InputToken.getPayload() instanceof String) {
	files.add(new PlaceholderFile((String) m_InputToken.getPayload()).getAbsoluteFile());
      }
      else if (m_InputToken.getPayload() instanceof String[]) {
	tmpStr = (String[]) m_InputToken.getPayload();
	for (i = 0; i < tmpStr.length; i++)
	  files.add(new PlaceholderFile(tmpStr[i]).getAbsoluteFile());
      }

      // add files
      if (isDebugOn())
	debug("Adding files: " + files);
      for (i = 0; i < files.size(); i++)
	exp.getDatasets().addElement(files.get(i));
    }

    // run experiment
    if (result == null) {
      try {
	if (isDebugOn())
	  debug("Initializing experiment...");
	exp.initialize();

	if (isDebugOn())
	  debug("Running experiment...");
	exp.runExperiment();

	if (isDebugOn())
	  debug("Post-processing experiment...");
	exp.postProcess();

	if (isDebugOn())
	  debug("Finished experiment!");

	m_OutputToken = new Token(exp);
      }
      catch (Exception e) {
	result = e.toString();
	getSystemErr().printStackTrace(e);
      }
    }

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->weka.experiment.Experiment.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{weka.experiment.Experiment.class};
  }
}
