/*
 * FileSelectorPanel.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.io.File;

import weka.core.converters.AbstractFileLoader;
import weka.core.converters.AbstractFileSaver;
import weka.core.converters.ConverterUtils;
import weka.gui.ConverterFileChooser;
import adams.core.io.PlaceholderFile;

/**
 * A panel that contains a text field with the current file and a
 * button for bringing up a ConverterFileChooser.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3499 $
 * @see ConverterFileChooser
 */
public class DatasetFileChooserPanel
  extends FileChooserPanel {

  /** for serialization. */
  private static final long serialVersionUID = -8755020252465094120L;

  /** the filechooser for selecting the dataset. */
  protected ConverterFileChooser m_FileChooser;

  /** the current loader. */
  protected AbstractFileLoader m_Loader;

  /** the current saver. */
  protected AbstractFileSaver m_Saver;

  /**
   * Initializes the panel with no file.
   */
  public DatasetFileChooserPanel() {
    this("");
  }

  /**
   * Initializes the panel with the given filename/directory.
   *
   * @param path	the filename/directory to use
   */
  public DatasetFileChooserPanel(String path) {
    this(new PlaceholderFile(path));
  }

  /**
   * Initializes the panel with the given filename/directory.
   *
   * @param path	the filename/directory to use
   */
  public DatasetFileChooserPanel(File path) {
    super(path);

    initializeConverters(path);
  }

  /**
   * Initializes the converters.
   *
   * @param path	the path/filename to use
   */
  protected void initializeConverters(File path) {
    if ((path.length() > 0) && path.isFile()) {
      try {
	m_Loader = ConverterUtils.getLoaderForFile(path.getAbsoluteFile());
	if (m_Loader != null)
	  m_Loader.setFile(path.getAbsoluteFile());
      }
      catch (Exception e) {
	e.printStackTrace();
      }
      m_Saver = ConverterUtils.getSaverForFile(path.getAbsoluteFile());
    }
    else {
      m_Loader = null;
      m_Saver  = null;
    }
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_FileChooser = new ConverterFileChooser();
    m_Loader      = null;
    m_Saver       = null;
  }

  /**
   * Performs the actual choosing of an object.
   *
   * @return		the chosen object or null if none chosen
   */
  protected File doChoose() {
    m_FileChooser.setSelectedFile(getCurrent().getAbsoluteFile());
    if (m_FileChooser.showOpenDialog(m_Self) == BaseFileChooser.APPROVE_OPTION) {
      m_Loader = m_FileChooser.getLoader();
      m_Saver  = m_FileChooser.getSaver();
      return new PlaceholderFile(m_FileChooser.getSelectedFile());
    }
    else {
      return null;
    }
  }

  /**
   * Returns the current loader. Only initialized after the user selected
   * a file with the filechooser.
   *
   * @return		the loader
   */
  public AbstractFileLoader getLoader() {
    if (m_Loader != null) {
      try {
	m_Loader.reset();
      }
      catch (Exception e) {
	e.printStackTrace();
      }
    }

    return m_Loader;
  }

  /**
   * Returns the current saver. Only initialized after the user selected
   * a file with the filechooser.
   *
   * @return		the saver
   */
  public AbstractFileSaver getSaver() {
    return m_Saver;
  }

  /**
   * Sets the current directory to use for the file chooser.
   *
   * @param value	the current directory
   */
  public void setCurrentDirectory(File value) {
    m_FileChooser.setCurrentDirectory(value.getAbsoluteFile());
  }

  /**
   * Returns the current directory in use by the file chooser.
   *
   * @return		the current directory
   */
  public File getCurrentDirectory() {
    return new PlaceholderFile(m_FileChooser.getCurrentDirectory());
  }

  /**
   * Sets the current value.
   *
   * @param value	the value to use, can be null
   */
  public boolean setCurrent(File value) {
    boolean	result;

    result = super.setCurrent(value);
    initializeConverters(getCurrent());
    m_FileChooser.setSelectedFile(getCurrent().getAbsoluteFile());

    return result;
  }
}
