/*
 * WekaInstanceViewer.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;

import adams.flow.core.Token;
import adams.gui.core.BasePanel;
import adams.gui.visualization.instance.InstanceContainer;
import adams.gui.visualization.instance.InstanceContainerManager;
import adams.gui.visualization.instance.InstancePanel;

/**
 <!-- globalinfo-start -->
 * Actor for displaying adams.data.instance.Instance objects in a graphical way (using the internal format), like the 'Instance Explorer' tool.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instances<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: InstanceViewer
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 640
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 480
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter [options]&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3948 $
 */
public class WekaInstanceViewer
  extends AbstractGraphicalDisplay
  implements DisplayPanelProvider, TextSupplier {

  /** for serialization. */
  private static final long serialVersionUID = 1283926389472133810L;

  /** the panel with the instances. */
  protected InstancePanel m_InstancePanel;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Actor for displaying adams.data.instance.Instance objects in a "
      + "graphical way (using the internal format), like the 'Instance Explorer' tool.";
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 800;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 500;
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    m_InstancePanel.getContainerManager().clear();
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    BasePanel	result;

    result          = new BasePanel(new BorderLayout());
    m_InstancePanel = new InstancePanel();
    result.add(m_InstancePanel, BorderLayout.CENTER);

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.core.Instances.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{weka.core.Instance.class, adams.data.instance.Instance.class};
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    InstanceContainerManager		manager;
    InstanceContainer			cont;
    adams.data.instance.Instance	inst;

    if (token.getPayload() instanceof weka.core.Instance) {
      inst = new adams.data.instance.Instance();
      inst.set((weka.core.Instance) token.getPayload());
    }
    else {
      inst = (adams.data.instance.Instance) token.getPayload();
    }

    manager = m_InstancePanel.getContainerManager();
    cont    = manager.newContainer(inst);
    manager.add(cont);
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    super.cleanUpGUI();

    if (m_InstancePanel != null)
      m_InstancePanel.getContainerManager().clear();
  }

  /**
   * Creates a new panel for the token.
   *
   * @param token	the token to display in a new panel
   * @return		the generated panel
   */
  public AbstractDisplayPanel createDisplayPanel(Token token) {
    AbstractDisplayPanel	result;

    result = new AbstractTextDisplayPanel("Instance Viewer") {
      private static final long serialVersionUID = -5618543590920864397L;
      protected InstancePanel m_InstancePanel;
      protected void initGUI() {
	super.initGUI();
	setLayout(new BorderLayout());
	m_InstancePanel = new InstancePanel();
	add(m_InstancePanel, BorderLayout.CENTER);
      }
      public void display(Token token) {
	adams.data.instance.Instance inst = null;
	if (token.getPayload() instanceof weka.core.Instance) {
	  inst = new adams.data.instance.Instance();
	  inst.set((weka.core.Instance) token.getPayload());
	}
	else {
	  inst = (adams.data.instance.Instance) token.getPayload();
	}
	InstanceContainerManager manager = m_InstancePanel.getContainerManager();
	InstanceContainer cont = manager.newContainer(inst);
	manager.add(cont);
      }
      public String supplyText() {
	return WekaInstanceViewer.supplyText(m_InstancePanel);
      }
      public void cleanUp() {
	m_InstancePanel.getContainerManager().clear();
      }
    };
    result.display(token);

    return result;
  }

  /**
   * Supplies the text.
   *
   * @return		the text, null if none available
   */
  public String supplyText() {
    return supplyText(m_InstancePanel);
  }

  /**
   * Returns the displayed instances as ARFF.
   *
   * @param panel	the panel to obtain the data form
   * @return		the generated ARFF content or null if no data available
   */
  protected static String supplyText(InstancePanel panel) {
    InstanceContainerManager 	manager;
    weka.core.Instances 	data;
    int				i;

    manager = panel.getContainerManager();
    if (manager.countVisible() == 0)
      return null;

    data = new weka.core.Instances(manager.getVisible(0).getData().getDatasetHeader());
    for (i = 0; i < manager.countVisible(); i++)
      data.add(manager.getVisible(i).getData().toInstance());

    return data.toString();
  }

  /**
   * Returns whether the created display panel requires a scroll pane or not.
   *
   * @return		true if the display panel requires a scroll pane
   */
  public boolean displayPanelRequiresScrollPane() {
    return true;
  }
}
