/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * WekaCapabilities.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;

import adams.flow.core.Capability;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Filters weka.core.Instance and weka.core.Instances based on defined capabilities. Only objects that match the capabilities will be passed on, all others get discarded.<br/>
 * The matching sense can be inverted as well.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instances<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.instance.Instance<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instances<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.instance.Instance<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Capabilities
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-capability &lt;NOMINAL_ATTRIBUTES|BINARY_ATTRIBUTES|UNARY_ATTRIBUTES|EMPTY_NOMINAL_ATTRIBUTES|NUMERIC_ATTRIBUTES|DATE_ATTRIBUTES|STRING_ATTRIBUTES|RELATIONAL_ATTRIBUTES|MISSING_VALUES|NO_CLASS|NOMINAL_CLASS|BINARY_CLASS|UNARY_CLASS|EMPTY_NOMINAL_CLASS|NUMERIC_CLASS|DATE_CLASS|STRING_CLASS|RELATIONAL_CLASS|MISSING_CLASS_VALUES|ONLY_MULTIINSTANCE&gt; [-capability ...] (property: capabilities)
 * &nbsp;&nbsp;&nbsp;The capabilities that the objects must match.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-invert (property: invert)
 * &nbsp;&nbsp;&nbsp;If set to true, then objects that failed the capabilities test will pass
 * &nbsp;&nbsp;&nbsp;through and all others get discarded.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class WekaCapabilities
  extends AbstractWekaInstanceAndWekaInstancesTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -6171057542176864122L;

  /** the class index. */
  protected Capability[] m_Capabilities;

  /** whether to invert the matching sense. */
  protected boolean m_Invert;

  /** the capabilities object to use. */
  protected weka.core.Capabilities m_ActualCapabilities;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Filters weka.core.Instance and weka.core.Instances based on defined "
      + "capabilities. Only objects that match the capabilities will be passed "
      + "on, all others get discarded.\n"
      + "The matching sense can be inverted as well.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "capability", "capabilities",
	    new Capability[]{});

    m_OptionManager.add(
	    "invert", "invert",
	    false);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_ActualCapabilities = null;
  }

  /**
   * Sets the capabilities.
   *
   * @param value	the capabilities
   */
  public void setCapabilities(Capability[] value) {
    m_Capabilities = value;
    reset();
  }

  /**
   * Returns the capabilities.
   *
   * @return		the capabilities
   */
  public Capability[] getCapabilities() {
    return m_Capabilities;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String capabilitiesTipText() {
    return "The capabilities that the objects must match.";
  }

  /**
   * Sets whether to invert the matching sense of the capabilities.
   *
   * @param value	if true then the matching sense gets inverted
   */
  public void setInvert(boolean value) {
    m_Invert = value;
    reset();
  }

  /**
   * Returns whether the matching sense of the capabilities is inverted.
   *
   * @return		true if the matching sense is inverted
   */
  public boolean getInvert() {
    return m_Invert;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String invertTipText() {
    return
        "If set to true, then objects that failed the capabilities test "
      + "will pass through and all others get discarded.";
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String					result;
    HashSet<Capability>				capabilities;
    HashSet<weka.core.Capabilities.Capability>	unwanted;
    Iterator<weka.core.Capabilities.Capability>	iter;
    weka.core.Capabilities.Capability		cap;

    result = super.setUp();

    if (result == null) {
      m_ActualCapabilities = new weka.core.Capabilities(null);

      // add capabilities
      capabilities = new HashSet<Capability>(Arrays.asList(m_Capabilities));
      for (Capability c: m_Capabilities)
	m_ActualCapabilities.enable(Capability.toWeka(c));

      // remove unwanted, implied capabilities
      unwanted = new HashSet<weka.core.Capabilities.Capability>();
      iter     = m_ActualCapabilities.capabilities();
      while (iter.hasNext()) {
	cap = iter.next();
	if (!capabilities.contains(Capability.toAdams(cap)))
	  unwanted.add(cap);
      }
      for (weka.core.Capabilities.Capability c: unwanted)
	m_ActualCapabilities.disable(c);

      if (isDebugOn())
	debug("Capabilites: " + m_ActualCapabilities);
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    Object		o;
    weka.core.Instances	inst;
    boolean		forward;

    result  = null;
    forward = false;

    // dataset
    o    = m_InputToken.getPayload();
    inst = null;
    if (o instanceof weka.core.Instances)
      inst = (weka.core.Instances) o;
    else if (o instanceof adams.data.instance.Instance)
      inst = ((adams.data.instance.Instance) o).getDatasetHeader();
    else if (o instanceof weka.core.Instance)
      inst = ((weka.core.Instance) o).dataset();
    else
      result = "Cannot handle object of type " + o.getClass().getName() + "!";

    if (result == null) {
      if (m_Invert)
	forward = !m_ActualCapabilities.test(inst);
      else
	forward = m_ActualCapabilities.test(inst);

      // output
      if (forward)
	m_OutputToken = new Token(m_InputToken.getPayload());
    }

    return result;
  }
}
