/*
 * Decompiled with CFR 0.152.
 */
package ai.djl.repository;

import ai.djl.repository.FilenameUtils;
import ai.djl.repository.JarRepository;
import ai.djl.repository.LocalRepository;
import ai.djl.repository.MRL;
import ai.djl.repository.Metadata;
import ai.djl.repository.RemoteRepository;
import ai.djl.repository.Repository;
import ai.djl.repository.RepositoryFactory;
import ai.djl.repository.SimpleRepository;
import ai.djl.repository.SimpleUrlRepository;
import ai.djl.repository.zoo.ModelLoader;
import ai.djl.repository.zoo.ModelZoo;
import ai.djl.util.ClassLoaderUtils;
import ai.djl.util.JsonUtils;
import ai.djl.util.Utils;
import com.google.gson.JsonParseException;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RepositoryFactoryImpl
implements RepositoryFactory {
    private static final Logger logger = LoggerFactory.getLogger(RepositoryFactoryImpl.class);
    private static final RepositoryFactory FACTORY = new RepositoryFactoryImpl();
    private static final Map<String, RepositoryFactory> REGISTRY = RepositoryFactoryImpl.init();
    private static final Pattern PATTERN = Pattern.compile("(.+)/([\\d.]+)(/(.*))?");

    RepositoryFactoryImpl() {
    }

    static RepositoryFactory getFactory() {
        return FACTORY;
    }

    @Override
    public Repository newInstance(String name, URI uri) {
        RepositoryFactory factory;
        String scheme = uri.getScheme();
        if (scheme == null) {
            scheme = "file";
        }
        if ((factory = REGISTRY.get(scheme)) != null) {
            return factory.newInstance(name, uri);
        }
        try {
            uri.toURL();
        }
        catch (MalformedURLException e) {
            throw new IllegalArgumentException("Malformed URL: " + uri, e);
        }
        if ("tfhub.dev".equals(uri.getHost().toLowerCase(Locale.ROOT))) {
            String path = uri.getPath();
            if (path.endsWith("/")) {
                path = path.substring(0, path.length() - 1);
            }
            path = "/tfhub-modules" + path + ".tar.gz";
            try {
                uri = new URI("https", null, "storage.googleapis.com", -1, path, null, null);
            }
            catch (URISyntaxException e) {
                throw new IllegalArgumentException("Failed to append query string: " + uri, e);
            }
            String[] tokens = path.split("/");
            String modelName = tokens[tokens.length - 2];
            return new SimpleUrlRepository(name, uri, modelName);
        }
        Path path = Paths.get(RepositoryFactoryImpl.parseFilePath(uri), new String[0]);
        String fileName = path.toFile().getName();
        if (FilenameUtils.isArchiveFile(fileName)) {
            fileName = FilenameUtils.getNamePart(fileName);
            return new SimpleUrlRepository(name, uri, fileName);
        }
        throw new IllegalArgumentException("============: " + uri);
    }

    @Override
    public Set<String> getSupportedScheme() {
        return REGISTRY.keySet();
    }

    static void registerRepositoryFactory(RepositoryFactory factory) {
        for (String scheme : factory.getSupportedScheme()) {
            REGISTRY.put(scheme, factory);
        }
    }

    private static Map<String, RepositoryFactory> init() {
        ConcurrentHashMap<String, RepositoryFactory> registry = new ConcurrentHashMap<String, RepositoryFactory>();
        registry.put("file", new LocalRepositoryFactory());
        registry.put("jar", new JarRepositoryFactory());
        registry.put("djl", new DjlRepositoryFactory());
        if (S3RepositoryFactory.findS3Fuse() != null) {
            registry.put("s3", new S3RepositoryFactory());
        }
        if (GcsRepositoryFactory.findGcsFuse() != null) {
            registry.put("gs", new GcsRepositoryFactory());
        }
        ServiceLoader<RepositoryFactory> factories = ServiceLoader.load(RepositoryFactory.class);
        for (RepositoryFactory factory : factories) {
            for (String scheme : factory.getSupportedScheme()) {
                registry.put(scheme, factory);
            }
        }
        return registry;
    }

    static String parseFilePath(URI uri) {
        String uriPath = uri.getPath();
        if (uriPath == null) {
            uriPath = uri.getSchemeSpecificPart();
        }
        if (uriPath.startsWith("file:")) {
            uriPath = uriPath.substring(5);
        }
        if (uriPath.startsWith("/") && System.getProperty("os.name").startsWith("Win")) {
            uriPath = uriPath.substring(1);
        }
        return uriPath;
    }

    private static String exec(String ... cmd) throws IOException, InterruptedException {
        String logOutput;
        Process exec = new ProcessBuilder(cmd).redirectErrorStream(true).start();
        try (InputStream is = exec.getInputStream();){
            logOutput = Utils.toString(is);
        }
        int exitCode = exec.waitFor();
        if (0 != exitCode) {
            logger.error("exit: {}, {}", (Object)exitCode, (Object)logOutput);
            throw new IOException("Failed to execute: [" + String.join((CharSequence)" ", cmd) + "]");
        }
        logger.debug("{}", (Object)logOutput);
        return logOutput;
    }

    private static boolean isMounted(String path) throws IOException, InterruptedException {
        String[] lines;
        String out = RepositoryFactoryImpl.exec("df");
        for (String line : lines = out.split("\\s")) {
            if (!line.trim().equals(path)) continue;
            logger.debug("Mount point already mounted");
            return true;
        }
        return false;
    }

    private static final class LocalRepositoryFactory
    implements RepositoryFactory {
        private LocalRepositoryFactory() {
        }

        @Override
        public Repository newInstance(String name, URI uri) {
            Path path = Paths.get(RepositoryFactoryImpl.parseFilePath(uri), new String[0]);
            if (Files.exists(path, new LinkOption[0]) && Files.isDirectory(path, new LinkOption[0])) {
                try {
                    if (Files.walk(path, new FileVisitOption[0]).anyMatch(f -> this.isLocalRepository(path, (Path)f))) {
                        logger.debug("Found local repository: {}", (Object)path);
                        return new LocalRepository(name, path.toUri(), path);
                    }
                }
                catch (IOException e) {
                    logger.warn("Failed locate metadata.json file, defaulting to simple", (Throwable)e);
                }
            }
            return new SimpleRepository(name, uri, path);
        }

        private boolean isLocalRepository(Path root, Path file) {
            boolean bl;
            block10: {
                if (!Files.isRegularFile(file, new LinkOption[0]) || root.equals(file.getParent())) {
                    return false;
                }
                if (!"metadata.json".equals(file.toFile().getName())) {
                    return false;
                }
                BufferedReader reader = Files.newBufferedReader(file);
                try {
                    Metadata metadata = (Metadata)JsonUtils.GSON.fromJson((Reader)reader, Metadata.class);
                    boolean bl2 = bl = metadata.getMetadataVersion() != null && metadata.getArtifacts() != null;
                    if (reader == null) break block10;
                }
                catch (Throwable throwable) {
                    try {
                        if (reader != null) {
                            try {
                                ((Reader)reader).close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (JsonParseException | IOException e) {
                        logger.warn("Invalid metadata.json file", e);
                        return false;
                    }
                }
                ((Reader)reader).close();
            }
            return bl;
        }

        @Override
        public Set<String> getSupportedScheme() {
            return Collections.singleton("file");
        }
    }

    private static final class JarRepositoryFactory
    implements RepositoryFactory {
        private JarRepositoryFactory() {
        }

        @Override
        public Repository newInstance(String name, URI uri) {
            URI realUri;
            URL u;
            String p = uri.getPath();
            if (p.startsWith("/")) {
                p = p.substring(1);
            }
            if ((u = ClassLoaderUtils.getContextClassLoader().getResource(p)) == null) {
                throw new IllegalArgumentException("Resource not found: " + uri);
            }
            try {
                realUri = u.toURI();
            }
            catch (URISyntaxException e) {
                throw new IllegalArgumentException("Resource not found: " + uri, e);
            }
            Path path = Paths.get(RepositoryFactoryImpl.parseFilePath(realUri), new String[0]);
            String fileName = path.toFile().getName();
            if (FilenameUtils.isArchiveFile(fileName)) {
                fileName = FilenameUtils.getNamePart(fileName);
            }
            return new JarRepository(name, uri, fileName, realUri);
        }

        @Override
        public Set<String> getSupportedScheme() {
            return Collections.singleton("jar");
        }
    }

    private static final class DjlRepositoryFactory
    implements RepositoryFactory {
        private DjlRepositoryFactory() {
        }

        @Override
        public Repository newInstance(String name, URI uri) {
            ModelZoo zoo;
            String queryString = uri.getQuery();
            URI djlUri = queryString != null ? URI.create("https://mlrepo.djl.ai/?" + queryString) : URI.create("https://mlrepo.djl.ai/");
            RemoteRepository repo = new RemoteRepository(name, djlUri);
            String groupId = uri.getHost();
            if (groupId == null) {
                throw new IllegalArgumentException("Invalid djl URL: " + uri);
            }
            String artifactId = RepositoryFactoryImpl.parseFilePath(uri);
            if (artifactId.startsWith("/")) {
                artifactId = artifactId.substring(1);
            }
            if (artifactId.isEmpty()) {
                throw new IllegalArgumentException("Invalid djl URL: " + uri);
            }
            String version = null;
            String artifactName = null;
            Matcher m = PATTERN.matcher(artifactId);
            if (m.matches()) {
                artifactId = m.group(1);
                version = m.group(2);
                artifactName = m.group(4);
            }
            if ((zoo = ModelZoo.getModelZoo(groupId)) == null) {
                throw new IllegalArgumentException("ModelZoo not found in classpath: " + groupId);
            }
            ModelLoader loader = zoo.getModelLoader(artifactId);
            if (loader == null) {
                throw new IllegalArgumentException("Invalid djl URL: " + uri);
            }
            MRL mrl = repo.model(loader.getApplication(), groupId, artifactId, version, artifactName);
            repo.addResource(mrl);
            return repo;
        }

        @Override
        public Set<String> getSupportedScheme() {
            return Collections.singleton("djl");
        }
    }

    static final class S3RepositoryFactory
    implements RepositoryFactory {
        S3RepositoryFactory() {
        }

        @Override
        public Repository newInstance(String name, URI uri) {
            try {
                Path path = S3RepositoryFactory.mount(uri);
                return new SimpleRepository(name, uri, path);
            }
            catch (IOException | InterruptedException e) {
                throw new IllegalArgumentException("Failed to mount s3 bucket", e);
            }
        }

        @Override
        public Set<String> getSupportedScheme() {
            return Collections.singleton("s3");
        }

        static String findS3Fuse() {
            String[] directories;
            if (System.getProperty("os.name").startsWith("Win")) {
                logger.debug("mount-s3 is not supported on Windows");
                return null;
            }
            String gcsFuse = Utils.getEnvOrSystemProperty("MOUNT_S3", "/usr/bin/mount-s3");
            if (Files.isRegularFile(Paths.get(gcsFuse, new String[0]), new LinkOption[0])) {
                return gcsFuse;
            }
            String path = System.getenv("PATH");
            for (String dir : directories = path.split(File.pathSeparator)) {
                Path file = Paths.get(dir, "mount-s3");
                if (!Files.isRegularFile(file, new LinkOption[0])) continue;
                return file.toAbsolutePath().toString();
            }
            return null;
        }

        private static Path mount(URI uri) throws IOException, InterruptedException {
            String bucket = uri.getHost();
            String prefix = uri.getPath();
            if (!prefix.isEmpty()) {
                prefix = prefix.substring(1);
            }
            Path dir = Utils.getCacheDir().toAbsolutePath().normalize();
            dir = dir.resolve("s3").resolve(Utils.hash(uri.toString()));
            String path = dir.toString();
            if (Files.isDirectory(dir, new LinkOption[0])) {
                if (RepositoryFactoryImpl.isMounted(path)) {
                    return dir.resolve(prefix);
                }
            } else {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            RepositoryFactoryImpl.exec(new String[]{S3RepositoryFactory.findS3Fuse(), bucket, path});
            return dir.resolve(prefix);
        }
    }

    static final class GcsRepositoryFactory
    implements RepositoryFactory {
        GcsRepositoryFactory() {
        }

        @Override
        public Repository newInstance(String name, URI uri) {
            try {
                Path path = GcsRepositoryFactory.mount(uri);
                return new SimpleRepository(name, uri, path);
            }
            catch (IOException | InterruptedException e) {
                throw new IllegalArgumentException("Failed to mount gs bucket", e);
            }
        }

        @Override
        public Set<String> getSupportedScheme() {
            return Collections.singleton("gs");
        }

        static String findGcsFuse() {
            String[] directories;
            if (System.getProperty("os.name").startsWith("Win")) {
                logger.debug("gcsfuse is not supported on Windows");
                return null;
            }
            String gcsFuse = Utils.getEnvOrSystemProperty("GCSFUSE", "/usr/bin/gcsfuse");
            if (Files.isRegularFile(Paths.get(gcsFuse, new String[0]), new LinkOption[0])) {
                return gcsFuse;
            }
            String path = System.getenv("PATH");
            for (String dir : directories = path.split(File.pathSeparator)) {
                Path file = Paths.get(dir, "gcsfuse");
                if (!Files.isRegularFile(file, new LinkOption[0])) continue;
                return file.toAbsolutePath().toString();
            }
            return null;
        }

        private static Path mount(URI uri) throws IOException, InterruptedException {
            String bucket = uri.getHost();
            String prefix = uri.getPath();
            if (!prefix.isEmpty()) {
                prefix = prefix.substring(1);
            }
            Path dir = Utils.getCacheDir().toAbsolutePath().normalize();
            dir = dir.resolve("gs").resolve(Utils.hash(uri.toString()));
            String path = dir.toString();
            if (Files.isDirectory(dir, new LinkOption[0])) {
                if (RepositoryFactoryImpl.isMounted(path)) {
                    return dir.resolve(prefix);
                }
            } else {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            RepositoryFactoryImpl.exec(new String[]{GcsRepositoryFactory.findGcsFuse(), "--implicit-dirs", bucket, path});
            return dir.resolve(prefix);
        }
    }
}

