/*
 * Decompiled with CFR 0.152.
 */
package jsat.linear;

import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import jsat.linear.IndexValue;
import jsat.linear.Matrix;
import jsat.linear.Vec;
import jsat.math.Function;
import jsat.math.IndexFunction;
import jsat.utils.DoubleList;
import jsat.utils.IndexTable;

public class SparseVector
extends Vec {
    private static final long serialVersionUID = 8591745505666264662L;
    private int length;
    protected int used;
    protected int[] indexes;
    protected double[] values;
    private Double sumCache = null;
    private Double varianceCache = null;
    private Double minCache = null;
    private Double maxCache = null;

    public SparseVector(int length) {
        this(length, 10);
    }

    public SparseVector(List<Double> vals) {
        this(vals.size());
        int z = 0;
        for (int i = 0; i < vals.size(); ++i) {
            if (vals.get(i) == 0.0) continue;
            if (z >= this.indexes.length) {
                this.indexes = Arrays.copyOf(this.indexes, this.indexes.length * 3 / 2);
                this.values = Arrays.copyOf(this.values, this.values.length * 3 / 2);
            }
            this.indexes[z] = i;
            this.values[z++] = vals.get(i);
        }
    }

    public SparseVector(int length, int capacity) {
        this(new int[capacity], new double[capacity], length, 0);
    }

    public SparseVector(int[] indexes, double[] values, int length, int used) {
        if (values.length != indexes.length) {
            throw new IllegalArgumentException("Index and Value arrays must have the same length, instead index was " + indexes.length + " and values was " + values.length);
        }
        if (used < 0 || used > length || used > values.length) {
            throw new IllegalArgumentException("Bad used value. Used must be in the range of 0 and min of values length (" + values.length + ") and array length (" + length + "), instead was given " + used);
        }
        if (length <= 0) {
            throw new IllegalArgumentException("Length of sparse vector must be positive, not " + length);
        }
        this.used = used;
        this.length = length;
        this.indexes = indexes;
        this.values = values;
    }

    public SparseVector(Vec toCopy) {
        this(toCopy.length(), toCopy.nnz());
        for (IndexValue iv : toCopy) {
            this.indexes[this.used] = iv.getIndex();
            this.values[this.used++] = iv.getValue();
        }
    }

    private void clearCaches() {
        this.sumCache = null;
        this.varianceCache = null;
        this.minCache = null;
        this.maxCache = null;
    }

    @Override
    public int length() {
        return this.length;
    }

    public void setLength(int length) {
        if (this.used > 0 && length < this.indexes[this.used - 1]) {
            throw new RuntimeException("Can not set the length to a value less then an index already in use");
        }
        this.length = length;
    }

    @Override
    public int nnz() {
        return this.used;
    }

    private void removeNonZero(int nzIndex) {
        for (int i = nzIndex + 1; i < this.used; ++i) {
            this.values[i - 1] = this.values[i];
            this.indexes[i - 1] = this.indexes[i];
        }
        --this.used;
    }

    @Override
    public void increment(int index, double val) {
        if (index > this.length - 1 || index < 0) {
            throw new IndexOutOfBoundsException("Can not access an index larger then the vector or a negative index");
        }
        if (val == 0.0) {
            return;
        }
        int location = Arrays.binarySearch(this.indexes, 0, this.used, index);
        if (location < 0) {
            this.insertValue(location, index, val);
        } else {
            int n = location;
            this.values[n] = this.values[n] + val;
            if (this.values[location] == 0.0) {
                this.removeNonZero(location);
            }
        }
    }

    @Override
    public double get(int index) {
        if (index > this.length - 1 || index < 0) {
            throw new ArithmeticException("Can not access an index larger then the vector or a negative index");
        }
        int location = Arrays.binarySearch(this.indexes, 0, this.used, index);
        if (location < 0) {
            return 0.0;
        }
        return this.values[location];
    }

    @Override
    public void set(int index, double val) {
        if (index > this.length() - 1 || index < 0) {
            throw new IndexOutOfBoundsException(index + " does not fit in [0," + this.length + ")");
        }
        this.clearCaches();
        int insertLocation = Arrays.binarySearch(this.indexes, 0, this.used, index);
        if (insertLocation >= 0) {
            if (val != 0.0) {
                this.values[insertLocation] = val;
            } else {
                this.removeNonZero(insertLocation);
            }
        } else if (val != 0.0) {
            this.insertValue(insertLocation, index, val);
        }
    }

    private void insertValue(int insertLocation, int index, double val) {
        insertLocation = -(insertLocation + 1);
        if (this.used == this.indexes.length) {
            int newIndexesSize = Math.max(Math.min(this.indexes.length * 2, Integer.MAX_VALUE), 8);
            this.indexes = Arrays.copyOf(this.indexes, newIndexesSize);
            this.values = Arrays.copyOf(this.values, newIndexesSize);
        }
        if (insertLocation < this.used) {
            System.arraycopy(this.indexes, insertLocation, this.indexes, insertLocation + 1, this.used - insertLocation);
            System.arraycopy(this.values, insertLocation, this.values, insertLocation + 1, this.used - insertLocation);
        }
        this.indexes[insertLocation] = index;
        this.values[insertLocation] = val;
        ++this.used;
    }

    @Override
    public Vec sortedCopy() {
        int i;
        IndexTable it = new IndexTable(DoubleList.unmodifiableView(this.values, this.used));
        double[] newValues = new double[this.used];
        int[] newIndecies = new int[this.used];
        int lessThanZero = 0;
        for (i = 0; i < this.used; ++i) {
            int origIndex = it.index(i);
            newValues[i] = this.values[origIndex];
            if (newValues[i] < 0.0) {
                ++lessThanZero;
            }
            newIndecies[i] = i;
        }
        for (i = lessThanZero; i < this.used; ++i) {
            newIndecies[i] = this.length - (this.used - lessThanZero) + (i - lessThanZero);
        }
        SparseVector sv = new SparseVector(this.length);
        sv.used = this.used;
        sv.values = newValues;
        sv.indexes = newIndecies;
        return sv;
    }

    public int getLastNonZeroIndex() {
        if (this.used == 0) {
            return -1;
        }
        return this.indexes[this.used - 1];
    }

    @Override
    public double min() {
        if (this.minCache != null) {
            return this.minCache;
        }
        double result = 0.0;
        for (int i = 0; i < this.used; ++i) {
            result = Math.min(result, this.values[i]);
        }
        this.minCache = result;
        return this.minCache;
    }

    @Override
    public double max() {
        if (this.maxCache != null) {
            return this.maxCache;
        }
        double result = 0.0;
        for (int i = 0; i < this.used; ++i) {
            result = Math.max(result, this.values[i]);
        }
        this.maxCache = result;
        return this.maxCache;
    }

    @Override
    public double sum() {
        if (this.sumCache != null) {
            return this.sumCache;
        }
        double sum = 0.0;
        double c = 0.0;
        for (int i = 0; i < this.used; ++i) {
            double d = this.values[i];
            double y = d - c;
            double t = sum + y;
            c = t - sum - y;
            sum = t;
        }
        this.sumCache = sum;
        return this.sumCache;
    }

    @Override
    public double variance() {
        if (this.varianceCache != null) {
            return this.varianceCache;
        }
        double mu = this.mean();
        double tmp = 0.0;
        double N = this.length();
        for (int i = 0; i < this.used; ++i) {
            tmp += Math.pow(this.values[i] - mu, 2.0);
        }
        tmp += (double)(this.length() - this.used) * Math.pow(0.0 - mu, 2.0);
        this.varianceCache = tmp /= N;
        return this.varianceCache;
    }

    @Override
    public double median() {
        if (this.used < this.length / 2) {
            return 0.0;
        }
        return super.median();
    }

    @Override
    public double skewness() {
        double mean = this.mean();
        double numer = 0.0;
        double denom = 0.0;
        for (int i = 0; i < this.used; ++i) {
            numer += Math.pow(this.values[i] - mean, 3.0);
            denom += Math.pow(this.values[i] - mean, 2.0);
        }
        numer += Math.pow(-mean, 3.0) * (double)(this.length - this.used);
        denom += Math.pow(-mean, 2.0) * (double)(this.length - this.used);
        double s1 = (numer /= (double)this.length) / Math.pow(denom /= (double)this.length, 1.5);
        if (this.length >= 3) {
            return Math.sqrt(this.length * (this.length - 1)) / (double)(this.length - 2) * s1;
        }
        return s1;
    }

    @Override
    public double kurtosis() {
        double mean = this.mean();
        double tmp = 0.0;
        double var = 0.0;
        for (int i = 0; i < this.used; ++i) {
            tmp += Math.pow(this.values[i] - mean, 4.0);
            var += Math.pow(this.values[i] - mean, 2.0);
        }
        tmp += Math.pow(-mean, 4.0) * (double)(this.length - this.used);
        var += Math.pow(-mean, 2.0) * (double)(this.length - this.used);
        return (tmp /= (double)this.length) / Math.pow(var /= (double)this.length, 2.0) - 3.0;
    }

    @Override
    public void copyTo(Vec destination) {
        if (destination instanceof SparseVector) {
            SparseVector other = (SparseVector)destination;
            if (other.indexes.length < this.used) {
                other.indexes = Arrays.copyOf(this.indexes, this.used);
                other.values = Arrays.copyOf(this.values, this.used);
                other.used = this.used;
                other.clearCaches();
            } else {
                other.used = this.used;
                other.clearCaches();
                System.arraycopy(this.indexes, 0, other.indexes, 0, this.used);
                System.arraycopy(this.values, 0, other.values, 0, this.used);
            }
        } else {
            super.copyTo(destination);
        }
    }

    @Override
    public double dot(Vec v) {
        double dot = 0.0;
        if (v instanceof SparseVector) {
            SparseVector b = (SparseVector)v;
            int p1 = 0;
            int p2 = 0;
            while (p1 < this.used && p2 < b.used) {
                int a1 = this.indexes[p1];
                int a2 = b.indexes[p2];
                if (a1 == a2) {
                    dot += this.values[p1++] * b.values[p2++];
                    continue;
                }
                if (a1 > a2) {
                    ++p2;
                    continue;
                }
                ++p1;
            }
        } else {
            if (v.isSparse()) {
                return super.dot(v);
            }
            for (int i = 0; i < this.used; ++i) {
                dot += this.values[i] * v.get(this.indexes[i]);
            }
        }
        return dot;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("[");
        int p = 0;
        for (int i = 0; i < this.length(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            if (p < this.used && this.indexes[p] == i) {
                sb.append(this.values[p++]);
                continue;
            }
            sb.append("0.0");
        }
        sb.append("]");
        return sb.toString();
    }

    @Override
    public void multiply(double c, Matrix A, Vec b) {
        if (this.length() != A.rows()) {
            throw new ArithmeticException("Vector x Matrix dimensions do not agree");
        }
        if (b.length() != A.cols()) {
            throw new ArithmeticException("Destination vector is not the right size");
        }
        for (int i = 0; i < this.used; ++i) {
            double val = c * this.values[i];
            int index = this.indexes[i];
            for (int j = 0; j < A.cols(); ++j) {
                b.increment(j, val * A.get(index, j));
            }
        }
    }

    @Override
    public void mutableAdd(double c) {
        if (c == 0.0) {
            return;
        }
        this.clearCaches();
        for (int i = 0; i < this.length(); ++i) {
            this.set(i, this.get(i) + c);
        }
    }

    @Override
    public void mutableAdd(double c, Vec v) {
        this.clearCaches();
        if (c == 0.0) {
            return;
        }
        if (v instanceof SparseVector) {
            SparseVector b = (SparseVector)v;
            int p1 = 0;
            int p2 = 0;
            while (p1 < this.used && p2 < b.used) {
                int a1 = this.indexes[p1];
                int a2 = b.indexes[p2];
                if (a1 == a2) {
                    int n = p1++;
                    this.values[n] = this.values[n] + c * b.values[p2];
                    ++p2;
                    continue;
                }
                if (a1 > a2) {
                    this.set(a2, c * b.values[p2]);
                    ++p1;
                    ++p2;
                    continue;
                }
                ++p1;
            }
            while (p2 < b.used) {
                this.set(b.indexes[p2], c * b.values[p2++]);
            }
        } else if (v.isSparse()) {
            if (v.nnz() == 0) {
                return;
            }
            int p1 = 0;
            Iterator<IndexValue> iter = v.getNonZeroIterator();
            IndexValue iv = iter.next();
            while (p1 < this.used && iv != null) {
                int a1 = this.indexes[p1];
                int a2 = iv.getIndex();
                if (a1 == a2) {
                    int n = p1++;
                    this.values[n] = this.values[n] + c * iv.getValue();
                    if (iter.hasNext()) {
                        iv = iter.next();
                        continue;
                    }
                    break;
                }
                if (a1 > a2) {
                    this.set(a2, c * iv.getValue());
                    ++p1;
                    if (iter.hasNext()) {
                        iv = iter.next();
                        continue;
                    }
                    break;
                }
                ++p1;
            }
        } else {
            for (int i = 0; i < this.length(); ++i) {
                this.set(i, this.get(i) + c * v.get(i));
            }
        }
    }

    @Override
    public void mutableMultiply(double c) {
        this.clearCaches();
        if (c == 0.0) {
            this.zeroOut();
            return;
        }
        int i = 0;
        while (i < this.used) {
            int n = i++;
            this.values[n] = this.values[n] * c;
        }
    }

    @Override
    public void mutableDivide(double c) {
        this.clearCaches();
        if (c == 0.0 && this.used != this.length) {
            throw new ArithmeticException("Division by zero would occur");
        }
        int i = 0;
        while (i < this.used) {
            int n = i++;
            this.values[n] = this.values[n] / c;
        }
    }

    @Override
    public double pNormDist(double p, Vec y) {
        if (this.length() != y.length()) {
            throw new ArithmeticException("Vectors must be of the same length");
        }
        double norm = 0.0;
        if (y instanceof SparseVector) {
            int p1 = 0;
            int p2 = 0;
            SparseVector b = (SparseVector)y;
            while (p1 < this.used && p2 < b.used) {
                int a1 = this.indexes[p1];
                int a2 = b.indexes[p2];
                if (a1 == a2) {
                    norm += Math.pow(Math.abs(this.values[p1] - b.values[p2]), p);
                    ++p1;
                    ++p2;
                    continue;
                }
                if (a1 > a2) {
                    norm += Math.pow(Math.abs(b.values[p2++]), p);
                    continue;
                }
                norm += Math.pow(Math.abs(this.values[p1++]), p);
            }
            while (p1 < this.used) {
                norm += Math.pow(Math.abs(this.values[p1++]), p);
            }
            while (p2 < b.used) {
                norm += Math.pow(Math.abs(b.values[p2++]), p);
            }
        } else {
            int z = 0;
            for (int i = 0; i < this.length(); ++i) {
                while (z < this.used && this.indexes[z] > i) {
                    norm += Math.pow(Math.abs(-y.get(i++)), p);
                }
                if (z < this.used && this.indexes[z] == i) {
                    norm += Math.pow(Math.abs(this.values[z++] - y.get(i)), p);
                    continue;
                }
                norm += Math.pow(Math.abs(-y.get(i)), p);
            }
        }
        return Math.pow(norm, 1.0 / p);
    }

    @Override
    public double pNorm(double p) {
        if (p <= 0.0) {
            throw new IllegalArgumentException("norm must be a positive value, not " + p);
        }
        double result = 0.0;
        if (p == 1.0) {
            for (int i = 0; i < this.used; ++i) {
                result += Math.abs(this.values[i]);
            }
        } else if (p == 2.0) {
            for (int i = 0; i < this.used; ++i) {
                result += this.values[i] * this.values[i];
            }
            result = Math.sqrt(result);
        } else if (Double.isInfinite(p)) {
            for (int i = 0; i < this.used; ++i) {
                result = Math.max(result, Math.abs(this.values[i]));
            }
        } else {
            for (int i = 0; i < this.used; ++i) {
                result += Math.pow(Math.abs(this.values[i]), p);
            }
            result = Math.pow(result, 1.0 / p);
        }
        return result;
    }

    @Override
    public SparseVector clone() {
        SparseVector copy = new SparseVector(this.length, Math.max(this.used, 10));
        System.arraycopy(this.values, 0, copy.values, 0, this.used);
        System.arraycopy(this.indexes, 0, copy.indexes, 0, this.used);
        copy.used = this.used;
        return copy;
    }

    @Override
    public void normalize() {
        double sum = 0.0;
        for (int i = 0; i < this.used; ++i) {
            sum += this.values[i] * this.values[i];
        }
        sum = Math.sqrt(sum);
        this.mutableDivide(Math.max(sum, 1.0E-10));
    }

    @Override
    public void mutablePairwiseMultiply(Vec b) {
        if (this.length() != b.length()) {
            throw new ArithmeticException("Vectors must have the same length");
        }
        this.clearCaches();
        for (int i = 0; i < this.used; ++i) {
            int n = i;
            this.values[n] = this.values[n] * b.get(this.indexes[i]);
        }
    }

    @Override
    public void mutablePairwiseDivide(Vec b) {
        if (this.length() != b.length()) {
            throw new ArithmeticException("Vectors must have the same length");
        }
        this.clearCaches();
        for (int i = 0; i < this.used; ++i) {
            int n = i;
            this.values[n] = this.values[n] / b.get(this.indexes[i]);
        }
    }

    @Override
    public boolean equals(Object obj, double range) {
        if (!(obj instanceof Vec)) {
            return false;
        }
        Vec otherVec = (Vec)obj;
        range = Math.abs(range);
        if (this.length() != otherVec.length()) {
            return false;
        }
        int z = 0;
        for (int i = 0; i < this.length(); ++i) {
            while (z < this.used && this.indexes[z] > i) {
                int n = i++;
                if (!(Math.abs(otherVec.get(n)) > range)) continue;
                return false;
            }
            if (z >= this.used || this.indexes[z] != i) continue;
            int n = z++;
            if (!(Math.abs(this.values[n] - otherVec.get(i)) > range)) continue;
            return Double.isNaN(this.values[z++]) && Double.isNaN(otherVec.get(i));
        }
        return true;
    }

    @Override
    public double[] arrayCopy() {
        double[] array = new double[this.length()];
        for (int i = 0; i < this.used; ++i) {
            array[this.indexes[i]] = this.values[i];
        }
        return array;
    }

    @Override
    public void applyFunction(Function f) {
        if (f.f(0.0) != 0.0) {
            super.applyFunction(f);
        } else {
            for (int i = 0; i < this.used; ++i) {
                this.values[i] = f.f(this.values[i]);
            }
        }
    }

    @Override
    public void applyIndexFunction(IndexFunction f) {
        if (f.f(0.0, -1.0) != 0.0) {
            super.applyIndexFunction(f);
        } else {
            int skip = 0;
            for (int i = 0; i < this.used; ++i) {
                this.indexes[i - skip] = this.indexes[i];
                this.values[i - skip] = f.indexFunc(this.values[i], i);
                if (this.values[i - skip] != 0.0) continue;
                ++skip;
            }
            this.used -= skip;
        }
    }

    @Override
    public void zeroOut() {
        this.used = 0;
    }

    @Override
    public Iterator<IndexValue> getNonZeroIterator(int start) {
        int tmpIndx;
        if (this.used <= 0) {
            return Collections.EMPTY_LIST.iterator();
        }
        final int startPos = start <= this.indexes[0] ? 0 : ((tmpIndx = Arrays.binarySearch(this.indexes, 0, this.used, start)) >= 0 ? tmpIndx : -tmpIndx - 1);
        Iterator<IndexValue> itor = new Iterator<IndexValue>(){
            int curUsedPos;
            IndexValue indexValue;
            {
                this.curUsedPos = startPos;
                this.indexValue = new IndexValue(-1, Double.NaN);
            }

            @Override
            public boolean hasNext() {
                return this.curUsedPos < SparseVector.this.used;
            }

            @Override
            public IndexValue next() {
                this.indexValue.setIndex(SparseVector.this.indexes[this.curUsedPos]);
                this.indexValue.setValue(SparseVector.this.values[this.curUsedPos++]);
                return this.indexValue;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException("Not supported yet.");
            }
        };
        return itor;
    }

    @Override
    public int hashCode() {
        int result = 1;
        for (int i = 0; i < this.used; ++i) {
            long bits = Double.doubleToLongBits(this.values[i]);
            result = 31 * result + (int)(bits ^ bits >>> 32);
            result = 31 * result + this.indexes[i];
        }
        return 31 * result + this.length;
    }

    @Override
    public boolean isSparse() {
        return true;
    }
}

