/*
 * Decompiled with CFR 0.152.
 */
package com.ongres.scram.common;

import com.ongres.scram.common.CryptoUtil;
import com.ongres.scram.common.StringPreparation;
import com.ongres.scram.common.exception.ScramRuntimeException;
import com.ongres.scram.common.util.Preconditions;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.SecretKeySpec;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

public enum ScramMechanism {
    SCRAM_SHA_1("SCRAM-SHA-1", "SHA-1", 160, "HmacSHA1", 4096),
    SCRAM_SHA_1_PLUS("SCRAM-SHA-1-PLUS", "SHA-1", 160, "HmacSHA1", 4096),
    SCRAM_SHA_224("SCRAM-SHA-224", "SHA-224", 224, "HmacSHA224", 4096),
    SCRAM_SHA_224_PLUS("SCRAM-SHA-224-PLUS", "SHA-224", 224, "HmacSHA224", 4096),
    SCRAM_SHA_256("SCRAM-SHA-256", "SHA-256", 256, "HmacSHA256", 4096),
    SCRAM_SHA_256_PLUS("SCRAM-SHA-256-PLUS", "SHA-256", 256, "HmacSHA256", 4096),
    SCRAM_SHA_384("SCRAM-SHA-384", "SHA-384", 384, "HmacSHA384", 4096),
    SCRAM_SHA_384_PLUS("SCRAM-SHA-384-PLUS", "SHA-384", 384, "HmacSHA384", 4096),
    SCRAM_SHA_512("SCRAM-SHA-512", "SHA-512", 512, "HmacSHA512", 10000),
    SCRAM_SHA_512_PLUS("SCRAM-SHA-512-PLUS", "SHA-512", 512, "HmacSHA512", 10000);

    private static final @Unmodifiable Map<String, ScramMechanism> BY_NAME_MAPPING;
    private static final @Unmodifiable List<String> SUPPORTED_MECHANISMS;
    @NotNull
    private final String mechanismName;
    @NotNull
    private final String hashAlgorithmName;
    private final int keyLength;
    @NotNull
    private final String hmacAlgorithmName;
    @NotNull
    private final String keyFactoryAlgorithmName;
    private final boolean channelBinding;
    private final int iterationCount;

    private ScramMechanism(String name, String hashAlgorithmName, int keyLength, String hmacAlgorithmName, int iterationCount) {
        this.mechanismName = Preconditions.checkNotNull(name, "name");
        this.hashAlgorithmName = Preconditions.checkNotNull(hashAlgorithmName, "hashAlgorithmName");
        this.keyLength = Preconditions.gt0(keyLength, "keyLength");
        this.hmacAlgorithmName = Preconditions.checkNotNull(hmacAlgorithmName, "hmacAlgorithmName");
        this.keyFactoryAlgorithmName = "PBKDF2With" + hmacAlgorithmName;
        this.channelBinding = name.endsWith("-PLUS");
        this.iterationCount = Preconditions.gt0(iterationCount, "iterationCount");
    }

    @NotNull
    String getHashAlgorithmName() {
        return this.hashAlgorithmName;
    }

    @NotNull
    String getHmacAlgorithmName() {
        return this.hmacAlgorithmName;
    }

    @NotNull
    public String getName() {
        return this.mechanismName;
    }

    public boolean isPlus() {
        return this.channelBinding;
    }

    int getKeyLength() {
        return this.keyLength;
    }

    int getIterationCount() {
        return this.iterationCount;
    }

    byte @NotNull [] digest(byte @NotNull [] message) {
        try {
            return MessageDigest.getInstance(this.hashAlgorithmName).digest(message);
        }
        catch (NoSuchAlgorithmException e) {
            throw new ScramRuntimeException("Hash algorithm " + this.hashAlgorithmName + " not present in current JVM", e);
        }
    }

    byte @NotNull [] hmac(byte @NotNull [] key, byte @NotNull [] message) {
        try {
            return CryptoUtil.hmac(new SecretKeySpec(key, this.hmacAlgorithmName), Mac.getInstance(this.hmacAlgorithmName), message);
        }
        catch (NoSuchAlgorithmException e) {
            throw new ScramRuntimeException("HMAC algorithm " + this.hmacAlgorithmName + " not present in current JVM", e);
        }
    }

    byte @NotNull [] saltedPassword(@NotNull StringPreparation stringPreparation, char @NotNull [] password, byte @NotNull [] salt, int iterationCount) {
        char[] normalizedPassword = stringPreparation.normalize(password);
        try {
            return CryptoUtil.hi(SecretKeyFactory.getInstance(this.keyFactoryAlgorithmName), this.keyLength, normalizedPassword, salt, iterationCount);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new ScramRuntimeException("Unsupported " + this.keyFactoryAlgorithmName + " for " + this.mechanismName, ex);
        }
    }

    @Nullable
    public static ScramMechanism byName(@NotNull String name) {
        return BY_NAME_MAPPING.get(Preconditions.checkNotNull(name, "name"));
    }

    @NotNull
    public static @Unmodifiable @NotNull List<@NotNull String> supportedMechanisms() {
        return Collections.unmodifiableList(SUPPORTED_MECHANISMS);
    }

    private static boolean isAlgorithmSupported(@NotNull ScramMechanism mechanism) {
        try {
            MessageDigest.getInstance(mechanism.hashAlgorithmName);
            Mac.getInstance(mechanism.hmacAlgorithmName);
            SecretKeyFactory.getInstance(mechanism.keyFactoryAlgorithmName);
            return true;
        }
        catch (NoSuchAlgorithmException e) {
            return false;
        }
    }

    static {
        BY_NAME_MAPPING = Arrays.stream(ScramMechanism.values()).filter(ScramMechanism::isAlgorithmSupported).collect(Collectors.collectingAndThen(Collectors.toMap(ScramMechanism::getName, Function.identity()), Collections::unmodifiableMap));
        SUPPORTED_MECHANISMS = BY_NAME_MAPPING.keySet().stream().collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
    }
}

