/*
 * Decompiled with CFR 0.152.
 */
package eu.verdelhan.ta4j;

import eu.verdelhan.ta4j.Decimal;
import eu.verdelhan.ta4j.Order;
import eu.verdelhan.ta4j.Strategy;
import eu.verdelhan.ta4j.Tick;
import eu.verdelhan.ta4j.TradingRecord;
import java.util.ArrayList;
import java.util.List;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadablePeriod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeSeries {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String name;
    private int beginIndex = -1;
    private int endIndex = -1;
    private final List<Tick> ticks;
    private Period timePeriod;
    private int maximumTickCount = Integer.MAX_VALUE;
    private int removedTicksCount = 0;
    private boolean subSeries = false;

    public TimeSeries(String name, List<Tick> ticks) {
        this(name, ticks, 0, ticks.size() - 1, false);
    }

    public TimeSeries(List<Tick> ticks) {
        this("unnamed", ticks);
    }

    public TimeSeries(String name, Period timePeriod) {
        if (timePeriod == null) {
            throw new IllegalArgumentException("Time period cannot be null");
        }
        this.name = name;
        this.ticks = new ArrayList<Tick>();
        this.timePeriod = timePeriod;
    }

    public TimeSeries(Period timePeriod) {
        this("unamed", timePeriod);
    }

    private TimeSeries(String name, List<Tick> ticks, int beginIndex, int endIndex, boolean subSeries) {
        if (endIndex < beginIndex - 1) {
            throw new IllegalArgumentException("end cannot be < than begin - 1");
        }
        this.name = name;
        this.ticks = ticks;
        this.beginIndex = beginIndex;
        this.endIndex = endIndex;
        this.subSeries = subSeries;
        this.computeTimePeriod();
    }

    public String getName() {
        return this.name;
    }

    public Tick getTick(int i) {
        int innerIndex = i - this.removedTicksCount;
        if (innerIndex < 0) {
            if (i < 0) {
                throw new IndexOutOfBoundsException(TimeSeries.buildOutOfBoundsMessage(this, i));
            }
            this.log.trace("Time series `{}` ({} ticks): tick {} already removed, use {}-th instead", new Object[]{this.name, this.ticks.size(), i, this.removedTicksCount});
            if (this.ticks.isEmpty()) {
                throw new IndexOutOfBoundsException(TimeSeries.buildOutOfBoundsMessage(this, this.removedTicksCount));
            }
            innerIndex = 0;
        } else if (innerIndex >= this.ticks.size()) {
            throw new IndexOutOfBoundsException(TimeSeries.buildOutOfBoundsMessage(this, i));
        }
        return this.ticks.get(innerIndex);
    }

    public Tick getFirstTick() {
        return this.getTick(this.beginIndex);
    }

    public Tick getLastTick() {
        return this.getTick(this.endIndex);
    }

    public int getTickCount() {
        if (this.endIndex < 0) {
            return 0;
        }
        int startIndex = Math.max(this.removedTicksCount, this.beginIndex);
        return this.endIndex - startIndex + 1;
    }

    public int getBegin() {
        return this.beginIndex;
    }

    public int getEnd() {
        return this.endIndex;
    }

    public String getSeriesPeriodDescription() {
        StringBuilder sb = new StringBuilder();
        if (!this.ticks.isEmpty()) {
            String timeFormat = "hh:mm dd/MM/yyyy";
            Tick firstTick = this.getFirstTick();
            Tick lastTick = this.getLastTick();
            sb.append(firstTick.getEndTime().toString("hh:mm dd/MM/yyyy")).append(" - ").append(lastTick.getEndTime().toString("hh:mm dd/MM/yyyy"));
        }
        return sb.toString();
    }

    public Period getTimePeriod() {
        return this.timePeriod;
    }

    public void setMaximumTickCount(int maximumTickCount) {
        if (this.subSeries) {
            throw new IllegalStateException("Cannot set a maximum tick count on a sub-series");
        }
        if (maximumTickCount <= 0) {
            throw new IllegalArgumentException("Maximum tick count must be strictly positive");
        }
        this.maximumTickCount = maximumTickCount;
        this.removeExceedingTicks();
    }

    public int getMaximumTickCount() {
        return this.maximumTickCount;
    }

    public int getRemovedTicksCount() {
        return this.removedTicksCount;
    }

    public void addTick(Tick tick) {
        if (tick == null) {
            throw new IllegalArgumentException("Cannot add null tick");
        }
        int lastTickIndex = this.ticks.size() - 1;
        if (!this.ticks.isEmpty()) {
            DateTime seriesEndTime = this.ticks.get(lastTickIndex).getEndTime();
            if (!tick.getEndTime().isAfter((ReadableInstant)seriesEndTime)) {
                throw new IllegalArgumentException("Cannot add a tick with end time <= to series end time");
            }
        }
        this.ticks.add(tick);
        if (this.beginIndex == -1) {
            this.beginIndex = 0;
        }
        ++this.endIndex;
        this.removeExceedingTicks();
    }

    public TimeSeries subseries(int beginIndex, int endIndex) {
        if (this.maximumTickCount != Integer.MAX_VALUE) {
            throw new IllegalStateException("Cannot create a sub-series from a time series for which a maximum tick count has been set");
        }
        return new TimeSeries(this.name, this.ticks, beginIndex, endIndex, true);
    }

    public TimeSeries subseries(int beginIndex, Period duration) {
        DateTime tickTime;
        DateTime beginInterval = this.getTick(beginIndex).getEndTime();
        DateTime endInterval = beginInterval.plus((ReadablePeriod)duration);
        Interval subseriesInterval = new Interval((ReadableInstant)beginInterval, (ReadableInstant)endInterval);
        int subseriesNbTicks = 0;
        for (int i = beginIndex; i <= this.endIndex && subseriesInterval.contains((ReadableInstant)(tickTime = this.getTick(i).getEndTime())); ++i) {
            ++subseriesNbTicks;
        }
        return this.subseries(beginIndex, beginIndex + subseriesNbTicks - 1);
    }

    public List<TimeSeries> split(int nbTicks) {
        ArrayList<TimeSeries> subseries = new ArrayList<TimeSeries>();
        for (int i = this.beginIndex; i <= this.endIndex; i += nbTicks) {
            int subseriesBegin = i;
            int subseriesEnd = Math.min(subseriesBegin + nbTicks - 1, this.endIndex);
            subseries.add(this.subseries(subseriesBegin, subseriesEnd));
        }
        return subseries;
    }

    public List<TimeSeries> split(Period splitDuration, Period sliceDuration) {
        ArrayList<TimeSeries> subseries = new ArrayList<TimeSeries>();
        if (splitDuration != null && !splitDuration.equals((Object)Period.ZERO) && sliceDuration != null && !sliceDuration.equals((Object)Period.ZERO)) {
            List<Integer> beginIndexes = this.getSplitBeginIndexes(splitDuration);
            for (Integer subseriesBegin : beginIndexes) {
                subseries.add(this.subseries((int)subseriesBegin, sliceDuration));
            }
        }
        return subseries;
    }

    public List<TimeSeries> split(Period duration) {
        return this.split(duration, duration);
    }

    public TradingRecord run(Strategy strategy) {
        return this.run(strategy, Order.OrderType.BUY);
    }

    public TradingRecord run(Strategy strategy, Order.OrderType orderType) {
        return this.run(strategy, orderType, Decimal.NaN);
    }

    public TradingRecord run(Strategy strategy, Order.OrderType orderType, Decimal amount) {
        int i;
        this.log.trace("Running strategy: {} (starting with {})", (Object)strategy, (Object)orderType);
        TradingRecord tradingRecord = new TradingRecord(orderType);
        for (i = this.beginIndex; i <= this.endIndex; ++i) {
            if (!strategy.shouldOperate(i, tradingRecord)) continue;
            tradingRecord.operate(i, this.ticks.get(i).getClosePrice(), amount);
        }
        if (!tradingRecord.isClosed()) {
            for (i = this.endIndex + 1; i < this.ticks.size(); ++i) {
                if (!strategy.shouldOperate(i, tradingRecord)) continue;
                tradingRecord.operate(i, this.ticks.get(i).getClosePrice(), amount);
                break;
            }
        }
        return tradingRecord;
    }

    private void computeTimePeriod() {
        Period minPeriod = null;
        for (int i = this.beginIndex; i < this.endIndex; ++i) {
            long currentPeriodMillis = this.getTick(i + 1).getEndTime().getMillis() - this.getTick(i).getEndTime().getMillis();
            if (minPeriod == null) {
                minPeriod = new Period(currentPeriodMillis);
                continue;
            }
            long minPeriodMillis = minPeriod.getMillis();
            if (minPeriodMillis <= currentPeriodMillis) continue;
            minPeriod = new Period(currentPeriodMillis);
        }
        if (minPeriod == null || Period.ZERO.equals(minPeriod)) {
            minPeriod = Period.days((int)1);
        }
        this.timePeriod = minPeriod;
    }

    private void removeExceedingTicks() {
        int tickCount = this.ticks.size();
        if (tickCount > this.maximumTickCount) {
            int nbTicksToRemove = tickCount - this.maximumTickCount;
            for (int i = 0; i < nbTicksToRemove; ++i) {
                this.ticks.remove(0);
            }
            this.removedTicksCount += nbTicksToRemove;
        }
    }

    private List<Integer> getSplitBeginIndexes(Period splitDuration) {
        ArrayList<Integer> beginIndexes = new ArrayList<Integer>();
        beginIndexes.add(this.beginIndex);
        DateTime beginInterval = this.getTick(this.beginIndex).getEndTime();
        DateTime endInterval = beginInterval.plus((ReadablePeriod)splitDuration);
        Interval splitInterval = new Interval((ReadableInstant)beginInterval, (ReadableInstant)endInterval);
        for (int i = this.beginIndex; i <= this.endIndex; ++i) {
            DateTime tickTime = this.getTick(i).getEndTime();
            if (splitInterval.contains((ReadableInstant)tickTime)) continue;
            if (!endInterval.isAfter((ReadableInstant)tickTime)) {
                beginIndexes.add(i);
            }
            beginInterval = endInterval.isBefore((ReadableInstant)tickTime) ? tickTime : endInterval;
            endInterval = beginInterval.plus((ReadablePeriod)splitDuration);
            splitInterval = new Interval((ReadableInstant)beginInterval, (ReadableInstant)endInterval);
        }
        return beginIndexes;
    }

    private static String buildOutOfBoundsMessage(TimeSeries series, int index) {
        return "Size of series: " + series.ticks.size() + " ticks, " + series.removedTicksCount + " ticks removed, index = " + index;
    }
}

