/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Wave.java
 * Copyright (C) 2018 University of Waikato, Hamilton, NZ
 */

package adams.flow.transformer.audiodata;

import adams.core.Utils;
import adams.core.base.BaseURL;
import adams.core.io.FileUtils;
import adams.data.audio.WaveContainer;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

/**
 * Reads data from WAV files or URLs.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 */
public class Wave
  extends AbstractAudioDataReader<WaveContainer> {

  private static final long serialVersionUID = -8099230864926096497L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Reads data from WAV files or URLs.";
  }

  /**
   * The accepted input types.
   *
   * @return		the input types
   */
  @Override
  public Class[] accepts() {
    return new Class[]{String.class, URL.class, BaseURL.class};
  }

  /**
   * The output generated by the reader.
   *
   * @return		the output type
   */
  @Override
  public Class generates() {
    return WaveContainer.class;
  }

  /**
   * Reads the data from the input.
   *
   * @param input	the input data
   * @return		the generated output
   * @throws Exception	if reading fails
   */
  @Override
  protected WaveContainer doRead(Object input) throws Exception {
    WaveContainer				result;
    InputStream					is;
    com.musicg.wave.Wave			wave;
    adams.flow.transformer.audioinfo.Wave	reader;
    Map<String,Object> 				info;
    Object					value;

    is = null;
    if (input instanceof String)
      is = new BufferedInputStream(new FileInputStream((String) input));
    else if (input instanceof URL)
      is = new BufferedInputStream(((URL) input).openStream());
    else if (input instanceof BaseURL)
      is = new BufferedInputStream(((BaseURL) input).urlValue().openStream());

    if (is != null) {
      wave   = new com.musicg.wave.Wave(is);
      result = new WaveContainer();
      result.setAudio(wave);
      FileUtils.closeQuietly(is);
      // source
      if (input instanceof String)
	result.getReport().setStringValue(WaveContainer.FILE, input.toString());
      else
	result.getReport().setStringValue(WaveContainer.URL, input.toString());
      // info
      reader = new adams.flow.transformer.audioinfo.Wave();
      info   = reader.read(wave);
      for (String key: info.keySet()) {
        value = info.get(key);
        if (value instanceof Boolean)
	  result.getReport().setBooleanValue(key, (Boolean) value);
        else if (value instanceof Number)
	  result.getReport().setNumericValue(key, ((Number) value).doubleValue());
        else
	  result.getReport().setStringValue(key, value.toString());
      }
      return result;
    }
    else {
      throw new IllegalStateException("Unhandled input data: " + Utils.classToString(input));
    }
  }
}
