/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractAudioDataReader.java
 * Copyright (C) 2018 University of Waikato, Hamilton, NZ
 */

package adams.flow.transformer.audiodata;

import adams.core.Utils;
import adams.core.option.AbstractOptionHandler;
import adams.data.audio.AbstractAudioContainer;
import adams.flow.core.Compatibility;

/**
 * Ancestor for readers for audio data.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @param <T> the output type
 */
public abstract class AbstractAudioDataReader<T extends AbstractAudioContainer>
  extends AbstractOptionHandler {

  private static final long serialVersionUID = -8842035286778396740L;

  /**
   * The accepted input types.
   *
   * @return		the input types
   */
  public abstract Class[] accepts();

  /**
   * The output generated by the reader.
   *
   * @return		the output type
   */
  public abstract Class generates();

  /**
   * Performs checks before reading from the input.
   *
   * @param input	the input to read
   * @return		null if successful, otherwise error message
   */
  protected String check(Object input) {
    Compatibility	comp;

    if (input == null)
      return "No input data provided!";
    comp = new Compatibility();
    if (!comp.isCompatible(new Class[]{input.getClass()}, accepts()))
      return "Expected " + Utils.classesToString(accepts()) + ", but received: " + Utils.classToString(input.getClass());
    return null;
  }

  /**
   * Reads the data from the input.
   *
   * @param input	the input data
   * @return		the generated output
   * @throws Exception	if reading fails
   */
  protected abstract T doRead(Object input) throws Exception;

  /**
   * Reads the data from the input.
   *
   * @param input	the input data
   * @return		the generated output
   * @throws Exception	if reading fails
   */
  public T read(Object input) throws Exception {
    String	msg;

    msg = check(input);
    if (msg != null)
      throw new IllegalStateException(msg);
    return doRead(input);
  }
}
