/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CNTKMultiFilterResultContainer.java
 * Copyright (C) 2018 University of Waikato, Hamilton, NZ
 */

package adams.flow.container;

import weka.core.Instances;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Container for storing all the output generated by the CNTKMultiFilter transformer.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 */
public class CNTKMultiFilterResultContainer
  extends AbstractContainer {

  private static final long serialVersionUID = 8175368852931742562L;

  /** the key for the dataset. */
  public final static String VALUE_DATASET = "Dataset";

  /** the key for the IDs JSON map. */
  public final static String VALUE_IDS = "IDs";

  /** the key for the saver commandline. */
  public final static String VALUE_SAVER = "Saver";

  /** the key for the CNTK JSON definition. */
  public final static String VALUE_DEFINITION = "Definition";

  /**
   * Default constructor.
   */
  public CNTKMultiFilterResultContainer() {
    super();
  }

  /**
   * Initializes the container.
   *
   * @param data	the converted data
   * @param ids		the dataset with the IDs (row id, sample ID)
   * @param saver 	the commandline for the saver
   * @param definition	the CNTK dimensions
   */
  public CNTKMultiFilterResultContainer(Instances data, String ids, String saver, String definition) {
    this();

    store(VALUE_DATASET,    data);
    store(VALUE_IDS,        ids);
    store(VALUE_SAVER,      saver);
    store(VALUE_DEFINITION, definition);
  }

  /**
   * Initializes the help strings.
   */
  protected void initHelp() {
    super.initHelp();

    addHelp(VALUE_DATASET,    "transformed dataset", Instances.class);
    addHelp(VALUE_DATASET,    "dataset with IDs (JSON Map; id <-> sample ID)", String.class);
    addHelp(VALUE_SAVER,      "Instances saver command-line", String.class);
    addHelp(VALUE_DEFINITION, "CNTK setup (JSON config file)", String.class);
  }

  /**
   * Returns all value names that can be used (theoretically).
   *
   * @return		iterator over all possible value names
   */
  @Override
  public Iterator<String> names() {
    List<String> result;

    result = new ArrayList<>();

    result.add(VALUE_DATASET);
    result.add(VALUE_IDS);
    result.add(VALUE_SAVER);
    result.add(VALUE_DEFINITION);

    return result.iterator();
  }

  /**
   * Checks whether the setup of the container is valid.
   *
   * @return		true if all the necessary values are available
   */
  @Override
  public boolean isValid() {
    return hasValue(VALUE_DATASET)
      && hasValue(VALUE_SAVER)
      && hasValue(VALUE_IDS)
      && hasValue(VALUE_DEFINITION);
  }
}
