/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Configurator.java
 * Copyright (C) 2017 University of Waikato, Hamilton, NZ
 */

package adams.ml.dl4j.modelgenerator;

import adams.core.QuickInfoHelper;
import adams.ml.dl4j.model.ModelConfigurator;
import adams.ml.dl4j.model.ModelWithScriptedConfiguration;
import org.deeplearning4j.nn.api.Model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Forwards model generated by configurator.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class Configurator
  extends AbstractModelGenerator {

  private static final long serialVersionUID = 2377758054389661470L;

  /** the model configurator. */
  protected ModelConfigurator m_ModelConfigurator;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Forwards the model generated by the " + ModelConfigurator.class.getName() + ".";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
      "model-configurator", "modelConfigurator",
      new ModelWithScriptedConfiguration());
  }

  /**
   * Sets the model configurator to use.
   *
   * @param value	the configurator
   */
  public void setModelConfigurator(ModelConfigurator value) {
    m_ModelConfigurator = value;
    reset();
  }

  /**
   * Returns the model configurator to use.
   *
   * @return 		the configurator
   */
  public ModelConfigurator getModelConfigurator() {
    return m_ModelConfigurator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String modelConfiguratorTipText() {
    return "The model configurator to use for generating the actual model.";
  }

  /**
   * Returns a quick info about the object, which can be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    return QuickInfoHelper.toString(this, "model", m_ModelConfigurator, "model: ");
  }

  /**
   * Generates the actual models.
   *
   * @param numInput	the number of input nodes
   * @param numOutput	the number of output nodes
   * @return		the models
   */
  protected List<Model> doGenerate(int numInput, int numOutput) {
    m_ModelConfigurator.setFlowContext(getFlowContext());
    return new ArrayList<>(Arrays.asList(m_ModelConfigurator.configureModel(numInput, numOutput)));
  }
}
