/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MultiGenerator.java
 * Copyright (C) 2017 University of Waikato, Hamilton, NZ
 */

package adams.ml.dl4j.modelgenerator;

import org.deeplearning4j.nn.api.Model;

import java.util.ArrayList;
import java.util.List;

/**
 * Combines models generated by specified generators.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class MultiGenerator
  extends AbstractModelGenerator {

  private static final long serialVersionUID = -901928613388638637L;

  /** the generators to use. */
  protected ModelGenerator[] m_Generators;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Combines the list of models generated by the specified generators.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
      "generator", "generators",
      new ModelGenerator[0]);
  }

  /**
   * Sets the model generators to use.
   *
   * @param value	the generators
   */
  public void setGenerators(ModelGenerator[] value) {
    m_Generators = value;
    reset();
  }

  /**
   * Returns the model configurator to use.
   *
   * @return 		the configurator
   */
  public ModelGenerator[] getGenerators() {
    return m_Generators;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String generatorsTipText() {
    return "The model generators to combine.";
  }

  /**
   * Generates the actual models.
   *
   * @param numInput	the number of input nodes
   * @param numOutput	the number of output nodes
   * @return		the models
   */
  @Override
  protected List<Model> doGenerate(int numInput, int numOutput) {
    List<Model>		result;
    int			i;

    result = new ArrayList<>();

    for (i = 0; i < m_Generators.length; i++) {
      if (isLoggingEnabled())
	getLogger().info("Generator #" + (i+1) + "...");
      m_Generators[i].setFlowContext(getFlowContext());
      result.addAll(m_Generators[i].generate(numInput, numOutput));
    }

    return result;
  }
}
