/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetRecordReaderConfigurator.java
 * Copyright (C) 2017 University of Waikato, Hamilton, NZ
 */

package adams.ml.dl4j.recordreader;

import adams.data.conversion.SpreadSheetToNumeric;
import adams.data.io.input.CsvSpreadSheetReader;
import adams.data.io.input.SpreadSheetReader;
import org.datavec.api.records.reader.RecordReader;

/**
 <!-- globalinfo-start -->
 * Allows reading of spreadsheets supported by ADAMS.
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * <pre>-logging-level &lt;OFF|SEVERE|WARNING|INFO|CONFIG|FINE|FINER|FINEST&gt; (property: loggingLevel)
 * &nbsp;&nbsp;&nbsp;The logging level for outputting errors and debugging output.
 * &nbsp;&nbsp;&nbsp;default: WARNING
 * </pre>
 * 
 * <pre>-input-split &lt;adams.ml.dl4j.inputsplit.InputSplitConfigurator&gt; (property: inputSplit)
 * &nbsp;&nbsp;&nbsp;The input split configurator to use for initializing the iterator.
 * &nbsp;&nbsp;&nbsp;default: adams.ml.dl4j.inputsplit.FileSplitConfigurator
 * </pre>
 * 
 * <pre>-reader &lt;adams.data.io.input.SpreadSheetReader&gt; (property: reader)
 * &nbsp;&nbsp;&nbsp;The spreadsheet reader to use.
 * &nbsp;&nbsp;&nbsp;default: adams.data.io.input.CsvSpreadSheetReader -data-row-type adams.data.spreadsheet.DenseDataRow -spreadsheet-type adams.data.spreadsheet.DefaultSpreadSheet
 * </pre>
 * 
 * <pre>-conversion &lt;adams.data.conversion.SpreadSheetToNumeric&gt; (property: conversion)
 * &nbsp;&nbsp;&nbsp;The conversion setup to use for converting non-numeric cells into numeric 
 * &nbsp;&nbsp;&nbsp;ones.
 * &nbsp;&nbsp;&nbsp;default: adams.data.conversion.SpreadSheetToNumeric -missing-value NaN -unhandled NaN
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class SpreadSheetRecordReaderConfigurator
  extends AbstractRecordReaderConfigurator {

  private static final long serialVersionUID = 8914456080710417165L;

  /** the reader to use. */
  protected SpreadSheetReader m_Reader;

  /** the conversion for turning non-numeric cells into numeric ones. */
  protected SpreadSheetToNumeric m_Conversion;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Allows reading of spreadsheets supported by ADAMS.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
      "reader", "reader",
      new CsvSpreadSheetReader());

    m_OptionManager.add(
      "conversion", "conversion",
      new SpreadSheetToNumeric());
  }

  /**
   * Sets the reader to use.
   *
   * @param value	the reader
   */
  public void setReader(SpreadSheetReader value) {
    m_Reader = value;
    reset();
  }

  /**
   * Returns the reader to use.
   *
   * @return 		the reader
   */
  public SpreadSheetReader getReader() {
    return m_Reader;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String readerTipText() {
    return "The spreadsheet reader to use.";
  }

  /**
   * Sets the conversion setup for turning non-numeric cells into numeric ones.
   *
   * @param value	the conversion
   */
  public void setConversion(SpreadSheetToNumeric value) {
    m_Conversion = value;
    reset();
  }

  /**
   * Returns the conversion setup for turning non-numeric cells into numeric ones.
   *
   * @return 		the conversion
   */
  public SpreadSheetToNumeric getConversion() {
    return m_Conversion;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String conversionTipText() {
    return "The conversion setup to use for converting non-numeric cells into numeric ones.";
  }

  /**
   * Configures the actual {@link RecordReader} and returns it.
   *
   * @return		the reader
   */
  @Override
  protected RecordReader doConfigureRecordReader() {
    return new SpreadSheetRecordReader(m_Reader, m_Conversion);
  }
}
