/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * GnuplotSpreadSheetWriter.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.Writer;
import java.util.Enumeration;

import adams.core.Range;
import adams.core.Utils;
import adams.core.io.SpreadSheet.Cell;
import adams.core.io.SpreadSheet.Row;
import adams.env.Environment;

/**
 <!-- globalinfo-start -->
 * Outputs all numeric columns of a spreadsheet in Gnuplot format.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-missing &lt;java.lang.String&gt; (property: missingValue)
 * &nbsp;&nbsp;&nbsp;The placeholder for missing values.
 * &nbsp;&nbsp;&nbsp;default: -999
 * </pre>
 *
 * <pre>-columns &lt;java.lang.String&gt; (property: columns)
 * &nbsp;&nbsp;&nbsp;The columns in the spreadsheet to output.
 * &nbsp;&nbsp;&nbsp;default: first-last
 * </pre>
 *
 * <pre>-invert (property: invert)
 * &nbsp;&nbsp;&nbsp;If set to true, then the matching sense is inverted.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 5563 $
 */
public class GnuplotSpreadSheetWriter
  extends AbstractSpreadSheetWriter {

  /** for serialization. */
  private static final long serialVersionUID = -1298185600402768643L;

  /** the line comment start. */
  public final static String COMMENT = "#";

  /** the default for missing values. */
  public final static String MISSING_VALUE = "-999";

  /** the range of columns to output in the data file. */
  protected Range m_Columns;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Outputs all numeric columns of a spreadsheet in Gnuplot format.";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Columns = new Range();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "columns",
	    new Range(Range.ALL));

    m_OptionManager.add(
	    "invert", "invert",
	    false);
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "Gnuplot data";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{"dat", "data"};
  }

  /**
   * Returns the default missing value.
   *
   * @return		the default for missing values
   */
  protected String getDefaultMissingValue() {
    return MISSING_VALUE;
  }

  /**
   * Sets the placeholder for missing values.
   *
   * @param value	the placeholder
   */
  public void setColumns(Range value) {
    m_Columns = value;
    reset();
  }

  /**
   * Returns the current placeholder for missing values.
   *
   * @return		the placeholder
   */
  public Range getColumns() {
    return m_Columns;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String columnsTipText() {
    return "The columns in the spreadsheet to output.";
  }

  /**
   * Sets whether to invert the matching sense.
   *
   * @param value	true if inverting matching sense
   */
  public void setInvert(boolean value) {
    m_Columns.setInverted(value);
    reset();
  }

  /**
   * Returns whether to invert the matching sense.
   *
   * @return		true if matching sense is inverted
   */
  public boolean getInvert() {
    return m_Columns.isInverted();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String invertTipText() {
    return "If set to true, then the matching sense is inverted.";
  }

  /**
   * Returns whether to write to an OutputStream rather than a Writer.
   *
   * @return		true if to write to an OutputStream
   */
  protected boolean getUseOutputStream() {
    return false;
  }

  /**
   * Performs the actual writing. The caller must ensure that the writer gets
   * closed.
   *
   * @param content	the spreadsheet to write
   * @param writer	the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, Writer writer) {
    boolean			result;
    Enumeration<String>		rowKeys;
    boolean			first;
    Row				row;
    Row				header;
    Cell			cell;
    int				i;
    int[]			indices;
    String			newline;
    boolean[]			isNumeric;
    String			colName;

    result = true;

    try {
      newline = System.getProperty("line.separator");

      m_Columns.setMax(content.getColumnCount());
      indices = m_Columns.getIntIndices();

      // determine numeric columns
      isNumeric = new boolean[indices.length];
      first     = true;
      for (i = 0; i < isNumeric.length; i++) {
	isNumeric[i] = content.isNumeric(indices[i]);
	if (isNumeric[i])
	  first = false;
      }
      // no numeric column found!
      if (first)
	return false;

      // generated by ADAMS
      writer.write(COMMENT + " Project: " + Environment.getInstance().getProject() + newline);
      writer.write(COMMENT + " User: " + System.getProperty("user.name") + newline);
      writer.write(COMMENT + newline);

      // spreadsheet name?
      if (content.hasName()) {
	writer.write(COMMENT + " " + content.getName() + newline);
	writer.write(COMMENT + newline);
      }

      // comments?
      if (content.getComments().size() > 0) {
	for (i = 0; i < content.getComments().size(); i++)
	  writer.write(COMMENT + " " + content.getComments().get(i) + newline);
	writer.write(COMMENT + newline);
      }

      // write header
      first = true;
      for (i = 0; i < indices.length; i++) {
	cell = content.getHeaderRow().getCell(indices[i]);

	// can we output cell?
	if (!isNumeric[i])
	  continue;

	if (first)
	  writer.write(COMMENT + " ");
	else
	  writer.write("\t");
	if (cell.isMissing())
	  writer.write(Utils.doubleQuote(m_MissingValue));
	else
	  writer.write(Utils.doubleQuote(cell.getContent()));

	first = false;
      }
      writer.write(newline);

      // write data rows
      header  = content.getHeaderRow();
      rowKeys = content.rowKeys();
      while (rowKeys.hasMoreElements()) {
	row      = content.getRow(rowKeys.nextElement());
	first    = true;
	for (i = 0; i < indices.length; i++) {
	  colName = header.getCellKey(indices[i]);
	  cell    = row.getCell(colName);

	  // can we output cell?
	  if (!isNumeric[i])
	    continue;

	  if (!first)
	    writer.write("\t");
	  if ((cell != null) && (cell.getContent() != null) && !cell.isMissing())
	    writer.write(Utils.doubleQuote(cell.getContent()));
	  else
	    writer.write(Utils.doubleQuote(m_MissingValue));

	  first = false;
	}
	writer.write(newline);
      }
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }
}
