/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * HeatmapStatistic.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.data.heatmap;

import java.util.Collections;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import adams.data.statistics.InformativeStatistic;

/**
 * Generates some statistics for a heatmap.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6027 $
 */
public class HeatmapStatistic
  implements InformativeStatistic {

  /** the width. */
  public static String WIDTH = "Map width";

  /** the height. */
  public static String HEIGHT = "Map height";

  /** the minimum value. */
  public static String MINIMUM = "Minimum";

  /** the maximum value. */
  public static String MAXIMUM = "Maximum";

  /** the zeroes count. */
  public static String COUNT_ZEROES = "Zeroes count";

  /** the non-zeroes count. */
  public static String COUNT_NONZEROES = "Non-zeroes count";

  /** the statistics. */
  protected Hashtable<String,Double> m_Statistics;

  /** the heatmap to create the statistics for. */
  protected Heatmap m_Heatmap;

  /**
   * Initializes the statistics.
   */
  public HeatmapStatistic() {
    super();

    m_Statistics = new Hashtable<String,Double>();
    m_Heatmap    = null;
  }

  /**
   * Initializes the statistics with the specified heatmap.
   *
   * @param map		the heatmap to generate the stats for
   */
  public HeatmapStatistic(Heatmap map) {
    this();
    setHeatmap(map);
  }

  /**
   * Sets the heatmap to generate the statistics for.
   *
   * @param value	the heatmap to use
   */
  public void setHeatmap(Heatmap value) {
    m_Heatmap = value;
    calculate();
  }

  /**
   * Returns the underlying heatmap.
   *
   * @return		the heatmap, null if none set
   */
  public Heatmap getHeatmap() {
    return m_Heatmap;
  }

  /**
   * Generates the statistics.
   */
  protected void calculate() {
    double	zeroes;
    double	nonZeroes;
    int		i;

    m_Statistics.clear();
    m_Statistics.put(HEIGHT, 0.0);
    m_Statistics.put(WIDTH, 0.0);
    m_Statistics.put(MINIMUM, 0.0);
    m_Statistics.put(MAXIMUM, 0.0);
    m_Statistics.put(COUNT_ZEROES, 0.0);
    m_Statistics.put(COUNT_NONZEROES, 0.0);

    if (m_Heatmap == null)
      return;

    zeroes    = 0.0;
    nonZeroes = 0.0;
    for (i = 0; i < m_Heatmap.size(); i++) {
      if (m_Heatmap.get(i) == 0.0)
	zeroes++;
      else
	nonZeroes++;
    }

    m_Statistics.put(HEIGHT, (double) m_Heatmap.getHeight());
    m_Statistics.put(WIDTH, (double) m_Heatmap.getWidth());
    m_Statistics.put(MINIMUM, m_Heatmap.getMin());
    m_Statistics.put(MAXIMUM, m_Heatmap.getMax());
    m_Statistics.put(COUNT_ZEROES, zeroes);
    m_Statistics.put(COUNT_NONZEROES, nonZeroes);
  }

  /**
   * Returns a description for this statistic.
   *
   * @return		the description
   */
  public String getStatisticDescription() {
    if (m_Heatmap == null)
      return "Statistics";
    else
      return m_Heatmap.getID();
  }

  /**
   * Returns all the names of the available statistical values.
   *
   * @return		the enumeration of names
   */
  public Enumeration<String> statisticNames() {
    Vector<String>	result;

    result = new Vector<String>(m_Statistics.keySet());
    Collections.sort(result);

    return result.elements();
  }

  /**
   * Returns the statistical value for the given statistic name.
   *
   * @param name	the name of the statistical value
   * @return		the corresponding value
   */
  public double getStatistic(String name) {
    return m_Statistics.get(name).doubleValue();
  }

  /**
   * Returns a string representation of the statistic.
   *
   * @return		the string representation
   */
  public String toString() {
    StringBuilder	result;
    Enumeration<String>	names;
    String		name;

    result = new StringBuilder();
    result.append(getStatisticDescription());
    result.append("\n");

    names = statisticNames();
    while (names.hasMoreElements()) {
      name = names.nextElement();
      result.append(name + ": " + getStatistic(name));
      result.append("\n");
    }

    return result.toString();
  }
}
