/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * MultiBoofCVFeatureGenerator.java
 * Copyright (C) 2014 University of Waikato, Hamilton, New Zealand
 */
package adams.data.boofcv.features;

import java.util.ArrayList;
import java.util.List;

import adams.core.Utils;
import adams.data.boofcv.BoofCVImageContainer;
import adams.data.featureconverter.HeaderDefinition;

/**
 <!-- globalinfo-start -->
 * Applies multiple generators to the same image and merges the generate a feature vectors side-by-side. If one of the generators should create fewer a feature vectors, missing values are used in that case.
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * <pre>-logging-level &lt;OFF|SEVERE|WARNING|INFO|CONFIG|FINE|FINER|FINEST&gt; (property: loggingLevel)
 * &nbsp;&nbsp;&nbsp;The logging level for outputting errors and debugging output.
 * &nbsp;&nbsp;&nbsp;default: WARNING
 * </pre>
 * 
 * <pre>-converter &lt;adams.data.featureconverter.AbstractFeatureConverter&gt; (property: converter)
 * &nbsp;&nbsp;&nbsp;The feature converter to use to produce the output data.
 * &nbsp;&nbsp;&nbsp;default: adams.data.featureconverter.SpreadSheetFeatureConverter -data-row-type adams.data.spreadsheet.DenseDataRow -spreadsheet-type adams.data.spreadsheet.SpreadSheet
 * </pre>
 * 
 * <pre>-field &lt;adams.data.report.Field&gt; [-field ...] (property: fields)
 * &nbsp;&nbsp;&nbsp;The fields to add to the output.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-notes &lt;adams.core.base.BaseString&gt; [-notes ...] (property: notes)
 * &nbsp;&nbsp;&nbsp;The notes to add as attributes to the generated data, eg 'PROCESS INFORMATION'
 * &nbsp;&nbsp;&nbsp;.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-sub-generator &lt;AbstractBoofCVFeatureGenerator&gt; [-sub-generator ...] (property: subGenerators)
 * &nbsp;&nbsp;&nbsp;The feature generators to apply to the image.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-prefix &lt;java.lang.String&gt; (property: prefix)
 * &nbsp;&nbsp;&nbsp;The prefix to use for disambiguating the attribute names of the datasets 
 * &nbsp;&nbsp;&nbsp;generated by the generators; use '#' for the 1-based index of the feature 
 * &nbsp;&nbsp;&nbsp;generator; example '#-'
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class MultiBoofCVFeatureGenerator
  extends AbstractBoofCVFeatureGenerator {

  /** for serialization. */
  private static final long serialVersionUID = -4136037171201268286L;
  
  /** the flatteners to use. */
  protected AbstractBoofCVFeatureGenerator[] m_SubGenerators;
  
  /** the prefix to use to disambiguate the attributes. */
  protected String m_Prefix;
  
  /** the sub-headers. */
  protected HeaderDefinition[] m_SubHeaders;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return 
	"Applies multiple generators to the same image and merges the "
	+ "generate a feature vectors side-by-side. If one of the "
	+ "generators should create fewer a feature vectors, missing "
	+ "values are used in that case.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "sub-generator", "subGenerators",
	    new AbstractBoofCVFeatureGenerator[0]);

    m_OptionManager.add(
	    "prefix", "prefix",
	    "");
  }

  /**
   * Sets the flatteners to use.
   *
   * @param value 	the flatteners
   */
  public void setSubGenerators(AbstractBoofCVFeatureGenerator[] value) {
    m_SubGenerators = value;
    reset();
  }

  /**
   * Returns the flatteners to use.
   *
   * @return 		the flatteners
   */
  public AbstractBoofCVFeatureGenerator[] getSubGenerators() {
    return m_SubGenerators;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String subGeneratorsTipText() {
    return "The feature generators to apply to the image.";
  }

  /**
   * Sets the prefix to user for disambiguating the attributes.
   *
   * @param value 	the prefix
   */
  public void setPrefix(String value) {
    m_Prefix = value;
    reset();
  }

  /**
   * Returns the prefix to use for disambiguating the attributes.
   *
   * @return 		the prefix
   */
  public String getPrefix() {
    return m_Prefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String prefixTipText() {
    return 
	"The prefix to use for disambiguating the attribute names of the "
	+ "datasets generated by the generators; use '#' for the 1-based "
	+ "index of the feature generator; example '#-'";
  }

  /**
   * Generates the prefix for the specified flattener.
   * 
   * @param index	the index of the flattener
   * @return		the prefix
   */
  protected String createPrefix(int index) {
    String	result;
    
    result = m_Prefix;
    result = result.replace("#", "" + (index + 1));
    
    return result;
  }
  
  /**
   * Creates the header from a template image.
   *
   * @param img		the image to act as a template
   * @return		the generated header
   */
  @Override
  public HeaderDefinition createHeader(BoofCVImageContainer img) {
    HeaderDefinition	result;
    int			i;
    int			n;
    String		name;
    
    m_SubHeaders = new HeaderDefinition[m_SubGenerators.length];
    for (i = 0; i < m_SubHeaders.length; i++) {
      m_SubHeaders[i] = m_SubGenerators[i].createHeader(img);
      m_SubHeaders[i] = m_SubGenerators[i].postProcessHeader(m_SubHeaders[i]);
    }

    // disambiguate the attribute names
    if (!m_Prefix.isEmpty()) {
      for (i = 0; i < m_SubHeaders.length; i++) {
	for (n = 0; n < m_SubHeaders[i].size(); n++) {
	  name = createPrefix(i) + m_SubHeaders[i].getName(n);
	  m_SubHeaders[i].rename(n, name);
	}
      }
    }
    
    result = new HeaderDefinition();
    if (m_SubHeaders.length > 0) {
      for (i = 0; i < m_SubHeaders.length; i++)
	result.add(m_SubHeaders[i]);
    }
    
    return result;
  }

  /**
   * Performs the actual feature generation.
   *
   * @param img		the image to process
   * @return		the generated features
   */
  @Override
  public List<Object>[] generateRows(BoofCVImageContainer img) {
    List<Object>[]	result;
    List<Object>[][]	sub;
    int			i;
    int			n;
    int			max;
    int			subMax;
    List<Object>	filler;
    
    // generate features
    sub = new List[m_SubGenerators.length][];
    for (i = 0; i < m_SubGenerators.length; i++)
      sub[i] = m_SubGenerators[i].postProcessRows(img, m_SubGenerators[i].generateRows(img));
    
    // fill with missing values, if necessary
    max = 0;
    for (i = 0; i < sub.length; i++)
      max = Math.max(max, sub[i].length);
    for (i = 0; i < sub.length; i++) {
      subMax = 0;
      for (n = 0; n < sub[i].length; n++)
	subMax = Math.max(subMax, sub[i][n].size());
      filler = new ArrayList<Object>();
      for (n = 0; n < subMax; n++)
	filler.add(null);
      sub[i] = (List<Object>[]) Utils.adjustArray(sub[i], max, filler);
    }

    // merge datasets
    if (sub.length > 0) {
      result = new List[max];
      for (n = 0; n < max; n++) {
	result[n] = new ArrayList<Object>();
	for (i = 0; i < sub.length; i++)
	  result[n].addAll(sub[i][n]);
      }
    }
    else {
      result    = new List[1];
      result[0] = new ArrayList<Object>();
    }
    
    return result;
  }
}
