/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Crop.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.data.jai.transformer;

import java.awt.image.BufferedImage;

import adams.data.jai.BufferedImageContainer;

/**
 <!-- globalinfo-start -->
 * Crops the image to specified width and height. Where the crop rectangle starts is defined by the X and Y position and the anchor.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-x &lt;int&gt; (property: X)
 * &nbsp;&nbsp;&nbsp;The horizontal pixel position.
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-y &lt;int&gt; (property: Y)
 * &nbsp;&nbsp;&nbsp;The vertical pixel position.
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the crop rectangle.
 * &nbsp;&nbsp;&nbsp;default: 100
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the crop rectangle.
 * &nbsp;&nbsp;&nbsp;default: 75
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-anchor &lt;TOP_LEFT|TOP_RIGHT|CENTER|BOTTOM_LEFT|BOTTOM_RIGHT&gt; (property: anchor)
 * &nbsp;&nbsp;&nbsp;Defines where to anchor the position on the crop rectangle.
 * &nbsp;&nbsp;&nbsp;default: TOP_LEFT
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4785 $
 */
public class Crop
  extends AbstractJAITransformer {

  /** for serialization. */
  private static final long serialVersionUID = 2959486760492196174L;

  /**
   * Enumeration for the crop rectangle anchor.\
   * 
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4785 $
   */
  public enum Anchor {
    /** top left corner. */
    TOP_LEFT,
    /** top right corner. */
    TOP_RIGHT,
    /** center. */
    CENTER,
    /** bottom left corner. */
    BOTTOM_LEFT,
    /** bottom right corner. */
    BOTTOM_RIGHT
  }
  
  /** the X position of the crop rectangle. */
  protected int m_X;
  
  /** the Y position of the crop rectangle. */
  protected int m_Y;
  
  /** the width of the crop rectangle. */
  protected int m_Width;
  
  /** the height of the crop rectangle. */
  protected int m_Height;
  
  /** where to anchor the position on the rectangle. */
  protected Anchor m_Anchor;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return 
	"Crops the image to specified width and height. Where the crop "
	+ "rectangle starts is defined by the X and Y position and the anchor.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"x", "X",
	1, 1, null);

    m_OptionManager.add(
	"y", "Y",
	1, 1, null);

    m_OptionManager.add(
	"width", "width",
	100, 1, null);

    m_OptionManager.add(
	"height", "height",
	75, 1, null);

    m_OptionManager.add(
	"anchor", "anchor",
	Anchor.TOP_LEFT);
  }

  /**
   * Sets the X position.
   *
   * @param value	the position
   */
  public void setX(int value) {
    if (value > 0) {
      m_X = value;
      reset();
    }
    else {
      getSystemErr().println("X has to be >0, provided: " + value);
    }
  }

  /**
   * Returns the X position.
   *
   * @return		the position
   */
  public int getX() {
    return m_X;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String XTipText() {
    return "The horizontal pixel position.";
  }

  /**
   * Sets the Y position.
   *
   * @param value	the position
   */
  public void setY(int value) {
    if (value > 0) {
      m_Y = value;
      reset();
    }
    else {
      getSystemErr().println("Y has to be >0, provided: " + value);
    }
  }

  /**
   * Returns the Y position.
   *
   * @return		the position
   */
  public int getY() {
    return m_Y;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String YTipText() {
    return "The vertical pixel position.";
  }

  /**
   * Sets the width of the crop rectangle.
   *
   * @param value	the width
   */
  public void setWidth(int value) {
    if (value > 0) {
      m_Width = value;
      reset();
    }
    else {
      getSystemErr().println("Width has to be >0, provided: " + value);
    }
  }

  /**
   * Returns the width of the crop rectangle.
   *
   * @return		the width
   */
  public int getWidth() {
    return m_Width;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String widthTipText() {
    return "The width of the crop rectangle.";
  }

  /**
   * Sets the height of the crop rectangle.
   *
   * @param value	the height
   */
  public void setHeight(int value) {
    if (value > 0) {
      m_Height = value;
      reset();
    }
    else {
      getSystemErr().println("Height has to be >0, provided: " + value);
    }
  }

  /**
   * Returns the height of the crop rectangle.
   *
   * @return		the height
   */
  public int getHeight() {
    return m_Height;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String heightTipText() {
    return "The height of the crop rectangle.";
  }

  /**
   * Sets where to anchor the position on the rectangle.
   *
   * @param value	the anchor
   */
  public void setAnchor(Anchor value) {
    m_Anchor = value;
    reset();
  }

  /**
   * Returns where to anchor the position on the rectangle.
   *
   * @return		the anchor
   */
  public Anchor getAnchor() {
    return m_Anchor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the gui
   */
  public String anchorTipText() {
    return "Defines where to anchor the position on the crop rectangle.";
  }

  /**
   * Performs no transformation at all, just returns the input.
   *
   * @param img		the image to process (can be modified, since it is a copy)
   * @return		the copy of the image
   */
  protected BufferedImageContainer[] doTransform(BufferedImageContainer img) {
    BufferedImageContainer[]	result;
    BufferedImage		image;
    int				x;
    int				y;
    int				leftOrig;
    int				topOrig;
    int				heightOrig;
    int				widthOrig;
    int				xOrig;
    int				yOrig;
    
    result    = new BufferedImageContainer[1];
    result[0] = (BufferedImageContainer) img.getHeader();
    
    // generate cropped image
    image = new BufferedImage(m_Width, m_Height, BufferedImage.TYPE_INT_ARGB);
    switch (m_Anchor) {
      case TOP_LEFT:
	leftOrig = m_X - 1;
	topOrig  = m_Y - 1;
	break;
      case TOP_RIGHT:
	leftOrig = m_X - 1 - m_Width;
	topOrig  = m_Y - 1;
	break;
      case BOTTOM_LEFT:
	leftOrig = m_X - 1;
	topOrig  = m_Y - 1 - m_Height;
	break;
      case BOTTOM_RIGHT:
	leftOrig = m_X - 1 - m_Width;
	topOrig  = m_Y - 1 - m_Height;
	break;
      case CENTER:
	leftOrig = m_X - 1 - m_Width / 2;
	topOrig  = m_Y - 1 - m_Height / 2;
	break;
      default:
	throw new IllegalStateException("Unhandled anchor: " + m_Anchor);
    }

    if (isDebugOn()) {
      debug("x=" + (m_X - 1) + ", y=" + (m_Y - 1) + ", width=" + m_Width + ", height=" + m_Height + ", anchor=" + m_Anchor);
      debug("  --> " + "leftOrig=" + leftOrig + ", topOrig=" + topOrig);
    }
    
    heightOrig = img.getHeight();
    widthOrig  = img.getWidth();
    
    for (y = 0; y < m_Height; y++) {
      yOrig = topOrig + y;
      if ((yOrig < 0) || (yOrig >= heightOrig))
	continue;
      for (x = 0; x < m_Width; x++) {
	xOrig = leftOrig + x;
	if ((xOrig < 0) || (xOrig >= widthOrig))
	  continue;
	image.setRGB(x, y, img.getImage().getRGB(xOrig, yOrig));
      }
    }
    
    result[0].setImage(image);
    
    return result;
  }
}
