/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * BufferedImageHelper.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;

/**
 * Helper class for BufferedImage objects.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 7150 $
 */
public class BufferedImageHelper {

  /**
   * Returns a copy of a BufferedImage object.
   * <p/>
   * Taken from
   * <a href="http://stackoverflow.com/questions/3514158/how-do-you-clone-a-bufferedimage/3514297#3514297" target="_blank">here</a>
   * (CC BY-SA 3.0).
   *
   * @param img		the image to copy
   */
  public static BufferedImage deepCopy(BufferedImage img) {
    ColorModel 		cm;
    boolean 		isAlphaPremultiplied;
    WritableRaster 	raster;

    cm = img.getColorModel();
    isAlphaPremultiplied = cm.isAlphaPremultiplied();
    raster = img.copyData(null);

    return new BufferedImage(cm, raster, isAlphaPremultiplied, null);
  }

  /**
   * Returns all the pixels of the image as an int array (row-wise).
   *
   * @param img		the image to get the pixels from
   * @return		the pixel array
   * @see		BufferedImage#getRGB(int, int)
   */
  public static int[] getPixels(BufferedImage img) {
    int[]	result;
    int		y;
    int		x;
    int		i;

    result = new int[img.getWidth() * img.getHeight()];
    i      = 0;
    for (y = 0; y < img.getHeight(); y++) {
      for (x = 0; x < img.getWidth(); x++) {
	result[i] = img.getRGB(x, y);
	i++;
      }
    }

    return result;
  }

  /**
   * Returns all the pixels of the image as an int array (row-wise) with the 
   * RGB(A) components as second dimension.
   *
   * @param img		the image to get the pixels from
   * @return		the pixel array
   * @see		BufferedImage#getRGB(int, int)
   */
  public static int[][] getRGBPixels(BufferedImage img) {
    int[][]	result;
    int		y;
    int		x;
    int		i;
    int		pixel;

    result = new int[img.getWidth() * img.getHeight()][4];
    i      = 0;
    for (y = 0; y < img.getHeight(); y++) {
      for (x = 0; x < img.getWidth(); x++) {
	pixel = img.getRGB(x, y);
	result[i][0] = (pixel >> 16) & 0xFF;  // R
	result[i][1] = (pixel >>  8) & 0xFF;  // G
	result[i][2] = (pixel >>  0) & 0xFF;  // B
	result[i][3] = (pixel >> 24) & 0xFF;  // A
	i++;
      }
    }

    return result;
  }

  /**
   * Returns the pixels of the image as an 2-D int array (row-wise).
   *
   * @param img		the image to get the pixels from
   * @return		the pixel array
   * @see		BufferedImage#getRGB(int, int)
   */
  public static int[][] getPixelRaster(BufferedImage img) {
    int[][]	result;
    int		y;
    int		x;

    result = new int[img.getHeight()][img.getWidth()];
    for (y = 0; y < img.getHeight(); y++) {
      for (x = 0; x < img.getWidth(); x++)
	result[y][x] = img.getRGB(x, y);
    }

    return result;
  }
}
