/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Image.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer.draw;

import java.awt.Graphics;
import java.awt.image.BufferedImage;

import adams.core.QuickInfoHelper;
import adams.data.image.AbstractImage;
import adams.flow.core.GlobalActorHelper;
import adams.flow.core.GlobalActorReference;

/**
 <!-- globalinfo-start -->
 * Draws the image obtained from a global actor at the specified location.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;The color of the pixel.
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 * 
 * <pre>-x &lt;int&gt; (property: X)
 * &nbsp;&nbsp;&nbsp;The X position of the top-left corner of the image (1-based).
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-y &lt;int&gt; (property: Y)
 * &nbsp;&nbsp;&nbsp;The Y position of the top-left corner of the image (1-based).
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-image-actor &lt;adams.flow.core.GlobalActorReference&gt; (property: imageActor)
 * &nbsp;&nbsp;&nbsp;The global actor to use for obtaining the image from.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6830 $
 */
public class Image
  extends AbstractDrawOperation {

  /** for serialization. */
  private static final long serialVersionUID = -1242368406478391978L;

  /** the X position of the text (1-based). */
  protected int m_X;

  /** the Y position of the text (1-based). */
  protected int m_Y;

  /** the global actor to get the image from. */
  protected GlobalActorReference m_ImageActor;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Draws the image obtained from a global actor at the specified location.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "x", "X",
	    1, 1, null);

    m_OptionManager.add(
	    "y", "Y",
	    1, 1, null);

    m_OptionManager.add(
	    "image-actor", "imageActor",
	    new GlobalActorReference());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;

    result  = QuickInfoHelper.toString(this, "X", m_X, "X: ");
    result += QuickInfoHelper.toString(this, "Y", m_Y, ", Y: ");
    result += QuickInfoHelper.toString(this, "imageActor", m_ImageActor, ", Image: ");
    
    return result;
  }

  /**
   * Sets the X position of the image (top-left corner).
   *
   * @param value	the position, 1-based
   */
  public void setX(int value) {
    if (value > 0) {
      m_X = value;
      reset();
    }
    else {
      getSystemErr().println("X must be >0, provided: " + value);
    }
  }

  /**
   * Returns the X position of the image (top-left corner).
   *
   * @return		the position, 1-based
   */
  public int getX() {
    return m_X;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String XTipText() {
    return "The X position of the top-left corner of the image (1-based).";
  }

  /**
   * Sets the Y position of the image (top-left corner).
   *
   * @param value	the position, 1-based
   */
  public void setY(int value) {
    if (value > 0) {
      m_Y = value;
      reset();
    }
    else {
      getSystemErr().println("Y must be >0, provided: " + value);
    }
  }

  /**
   * Returns the Y position of the image (top-left corner).
   *
   * @return		the position, 1-based
   */
  public int getY() {
    return m_Y;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String YTipText() {
    return "The Y position of the top-left corner of the image (1-based).";
  }

  /**
   * Sets the global actor to obtain the image from.
   *
   * @param value	the actor reference
   */
  public void setImageActor(GlobalActorReference value) {
    m_ImageActor = value;
    reset();
  }

  /**
   * Returns the global actor to obtain the image from.
   *
   * @return		the actor reference
   */
  public GlobalActorReference getImageActor() {
    return m_ImageActor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String imageActorTipText() {
    return "The global actor to use for obtaining the image from.";
  }

  /**
   * Performs the actual draw operation.
   * 
   * @param image	the image to draw on
   */
  @Override
  protected String doDraw(BufferedImage image) {
    String		result;
    Graphics		g;
    Object		obj;
    BufferedImage	todraw;

    result = null;

    if (m_X > image.getWidth())
      result = "X is larger than image width: " + m_X + " > " + image.getWidth();
    else if (m_Y > image.getHeight())
      result = "Y is larger than image height: " + m_Y + " > " + image.getHeight();

    obj    = GlobalActorHelper.getSetupFromSource(Object.class, m_ImageActor, m_Owner);
    todraw = null;
    if (obj instanceof BufferedImage)
      todraw = (BufferedImage) obj;
    else if (obj instanceof AbstractImage)
      todraw = ((AbstractImage) obj).toBufferedImage();
    else
      result = "Unknown image class: " + obj.getClass().getName();
    
    if ((result == null) && (todraw != null)) {
      g = image.getGraphics();
      g.drawImage(todraw, m_X, m_Y, null);
    }
    
    return result;
  }
}
