/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * OpexObjectLocationsReader.java
 * Copyright (C) 2021-2023 University of Waikato, Hamilton, New Zealand
 */

package adams.data.io.input;

import adams.data.report.DataType;
import adams.data.report.Field;
import adams.data.report.Report;
import adams.flow.transformer.locateobjects.LocatedObject;
import adams.flow.transformer.locateobjects.LocatedObjects;
import adams.flow.transformer.locateobjects.ObjectPrefixHandler;
import opex4j.ObjectPrediction;
import opex4j.ObjectPredictions;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

/**
 <!-- globalinfo-start -->
 * Reads object locations from JSON generated by the opex library.<br>
 * <br>
 * See more:<br>
 * https:&#47;&#47;github.com&#47;WaikatoLink2020&#47;objdet-predictions-exchange-format
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * <pre>-logging-level &lt;OFF|SEVERE|WARNING|INFO|CONFIG|FINE|FINER|FINEST&gt; (property: loggingLevel)
 * &nbsp;&nbsp;&nbsp;The logging level for outputting errors and debugging output.
 * &nbsp;&nbsp;&nbsp;default: WARNING
 * </pre>
 *
 * <pre>-input &lt;adams.core.io.PlaceholderFile&gt; (property: input)
 * &nbsp;&nbsp;&nbsp;The file to read and turn into a report.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 *
 * <pre>-id &lt;adams.data.report.Field&gt; (property: ID)
 * &nbsp;&nbsp;&nbsp;The field to use for storing the ID.
 * &nbsp;&nbsp;&nbsp;default: ID[S]
 * </pre>
 *
 * <pre>-timestamp &lt;adams.data.report.Field&gt; (property: timestamp)
 * &nbsp;&nbsp;&nbsp;The field to use for storing the timestamp.
 * &nbsp;&nbsp;&nbsp;default: Timestamp[S]
 * </pre>
 *
 * <pre>-prefix &lt;java.lang.String&gt; (property: prefix)
 * &nbsp;&nbsp;&nbsp;The report field prefix used in the report for the objects.
 * &nbsp;&nbsp;&nbsp;default: Object.
 * </pre>
 *
 * <pre>-label-suffix &lt;java.lang.String&gt; (property: labelSuffix)
 * &nbsp;&nbsp;&nbsp;The suffix to use in the report for labels.
 * &nbsp;&nbsp;&nbsp;default: type
 * </pre>
 *
 * <pre>-score-suffix &lt;java.lang.String&gt; (property: scoreSuffix)
 * &nbsp;&nbsp;&nbsp;The suffix to use in the report for scores.
 * &nbsp;&nbsp;&nbsp;default: score
 * </pre>
 *
 * <pre>-meta-prefix &lt;java.lang.String&gt; (property: metaPrefix)
 * &nbsp;&nbsp;&nbsp;The report field prefix used in the report for the meta-data.
 * &nbsp;&nbsp;&nbsp;default: Meta.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author fracpete (fracpete at waikato dot ac dot nz)
 */
public class OpexObjectLocationsReader
  extends AbstractReportReader<Report>
  implements ObjectPrefixHandler, StringReportReader<Report> {

  private static final long serialVersionUID = -1069346880458474698L;

  /** the field to use for the ID. */
  protected Field m_ID;

  /** the field to use for the timestamp. */
  protected Field m_Timestamp;

  /** the prefix to use for objects. */
  protected String m_Prefix;

  /** the label suffix to use. */
  protected String m_LabelSuffix;

  /** the score suffix to use. */
  protected String m_ScoreSuffix;

  /** the prefix for the meta-data. */
  protected String m_MetaPrefix;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Reads object locations from JSON generated by the opex library.\n\n"
      + "See more:\n"
      + "https://github.com/WaikatoLink2020/objdet-predictions-exchange-format";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
      "id", "ID",
      new Field("ID", DataType.STRING));

    m_OptionManager.add(
      "timestamp", "timestamp",
      new Field("Timestamp", DataType.STRING));

    m_OptionManager.add(
      "prefix", "prefix",
      LocatedObjects.DEFAULT_PREFIX);

    m_OptionManager.add(
      "label-suffix", "labelSuffix",
      "type");

    m_OptionManager.add(
      "score-suffix", "scoreSuffix",
      "score");

    m_OptionManager.add(
      "meta-prefix", "metaPrefix",
      "Meta.");
  }

  /**
   * Sets the field to use for the ID.
   *
   * @param value 	the field
   */
  public void setID(Field value) {
    m_ID = value;
    reset();
  }

  /**
   * Returns the field to use for the ID.
   *
   * @return 		the field
   */
  public Field getID() {
    return m_ID;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String IDTipText() {
    return "The field to use for storing the ID.";
  }

  /**
   * Sets the field to use for the timestamp.
   *
   * @param value 	the field
   */
  public void setTimestamp(Field value) {
    m_Timestamp = value;
    reset();
  }

  /**
   * Returns the field to use for the timestamp.
   *
   * @return 		the field
   */
  public Field getTimestamp() {
    return m_Timestamp;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String timestampTipText() {
    return "The field to use for storing the timestamp.";
  }

  /**
   * Sets the field prefix used in the report for the objects.
   *
   * @param value 	the field prefix
   */
  @Override
  public void setPrefix(String value) {
    m_Prefix = value;
    reset();
  }

  /**
   * Returns the field prefix used in the report for the objects.
   *
   * @return 		the field prefix
   */
  @Override
  public String getPrefix() {
    return m_Prefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  @Override
  public String prefixTipText() {
    return "The report field prefix used in the report for the objects.";
  }

  /**
   * Sets the field suffix used in the report for labels.
   *
   * @param value 	the field suffix
   */
  public void setLabelSuffix(String value) {
    m_LabelSuffix = value;
    reset();
  }

  /**
   * Returns the field suffix used in the report for labels.
   *
   * @return 		the field suffix
   */
  public String getLabelSuffix() {
    return m_LabelSuffix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String labelSuffixTipText() {
    return "The suffix to use in the report for labels.";
  }

  /**
   * Sets the field suffix used in the report for scores.
   *
   * @param value 	the field suffix
   */
  public void setScoreSuffix(String value) {
    m_ScoreSuffix = value;
    reset();
  }

  /**
   * Returns the field suffix used in the report for scores.
   *
   * @return 		the field suffix
   */
  public String getScoreSuffix() {
    return m_ScoreSuffix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String scoreSuffixTipText() {
    return "The suffix to use in the report for scores.";
  }

  /**
   * Sets the field prefix used in the report for the meta-data.
   *
   * @param value 	the field prefix
   */
  public void setMetaPrefix(String value) {
    m_MetaPrefix = value;
    reset();
  }

  /**
   * Returns the field prefix used in the report for the meta-data.
   *
   * @return 		the field prefix
   */
  public String getMetaPrefix() {
    return m_MetaPrefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String metaPrefixTipText() {
    return "The report field prefix used in the report for the meta-data.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  @Override
  public String getFormatDescription() {
    return "Opex annotations";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension(s) (without the dot!)
   */
  @Override
  public String[] getFormatExtensions() {
    return new String[]{"json"};
  }

  /**
   * Tries to determine the parent ID for the current report.
   *
   * @param report	the report to determine the ID for
   * @return		the parent database ID, -1 if it cannot be determined
   */
  @Override
  protected int determineParentID(Report report) {
    return -1;
  }

  /**
   * Returns a new instance of the report class in use.
   *
   * @return		the new (empty) report
   */
  @Override
  public Report newInstance() {
    return new Report();
  }

  /**
   * Converts the predictions into a report.
   *
   * @param preds	the predictions to convert
   * @return		the reports that were read
   */
  protected List<Report> readData(ObjectPredictions preds) {
    List<Report> 	result;
    LocatedObject 	lobj;
    LocatedObjects 	lobjs;
    Report		report;

    result = new ArrayList<>();
    report = newInstance();
    report.addField(m_ID);
    report.addField(m_Timestamp);

    // meta-data
    if (preds.getTimestamp() != null)
      report.setStringValue(m_Timestamp.getName(), ObjectPredictions.TIMESTAMP_FORMATTER.format(preds.getTimestamp()));
    report.setStringValue(m_ID.getName(), preds.getID());
    for (String key: preds.getMeta().keySet())
      report.setStringValue(m_MetaPrefix + key, "" + preds.getMeta().get(key));

    // objects
    lobjs = new LocatedObjects();
    for (ObjectPrediction pred: preds.getObjects()) {
      lobj = new LocatedObject(pred.getBBox().toRectangle());
      lobj.setPolygon(pred.getPolygon().toPolygon());
      lobj.getMetaData().put(m_LabelSuffix, pred.getLabel());
      if (pred.getScore() != null)
	lobj.getMetaData().put(m_ScoreSuffix, pred.getScore());
      lobj.getMetaData().putAll(pred.getMeta());
      lobjs.add(lobj);
    }

    report.mergeWith(lobjs.toReport(m_Prefix));
    result.add(report);

    return result;
  }

  /**
   * Performs the actual reading.
   *
   * @return		the reports that were read
   */
  @Override
  protected List<Report> readData() {
    List<Report> 	result;

    result = new ArrayList<>();
    try {
      result = readData(ObjectPredictions.newInstance(m_Input.getAbsoluteFile()));
    }
    catch (Exception e) {
      getLogger().log(Level.SEVERE, "Failed to read JSON file: " + m_Input, e);
    }

    return result;
  }

  /**
   * Reads the data.
   *
   * @param s the string to read from
   * @return the report loaded from the string
   */
  @Override
  public List<Report> read(String s) {
    List<Report> 	result;

    result = new ArrayList<>();
    try {
      result = readData(ObjectPredictions.newInstance(s));
    }
    catch (Exception e) {
      getLogger().log(Level.SEVERE, "Failed to read JSON string: " + s, e);
    }

    return result;
  }
}
