/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * LatexProducer.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.util.Date;

import adams.core.ClassLocator;
import adams.core.Utils;
import adams.core.io.FileUtils;
import adams.env.Environment;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorUtils;

/**
 * Generates the JSON format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 5846 $
 */
public class LatexProducer
  extends AbstractRecursiveOptionProducer<String,StringBuilder>
  implements FileBasedProducer {

  /** for serialization. */
  private static final long serialVersionUID = 6207182996484984270L;

  /** whether to print in landscape. */
  protected boolean m_Landscape;
  
  /** for storing the structure. */
  protected StringBuilder m_Structure;
  
  /** for storing the content. */
  protected StringBuilder m_Details;

  /** the index in the structure. */
  protected int m_Index;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return
        "Generates LaTeX documentation.";
  }

  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_UsePropertyNames = true;
  }

  /**
   * Sets whether to use portrait or landscape.
   *
   * @param value 	true if to use landscape
   */
  public void setLandscape(boolean value) {
    m_Landscape = value;
  }

  /**
   * Returns whether to use portrait or landscape
   *
   * @return 		true if landscape in use
   */
  public boolean getLandscape() {
    return m_Landscape;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String landscapeTipText() {
    return "Whether to generate a document in landscape format.";
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  @Override
  protected String initOutput() {
    return new String();
  }

  /**
   * Adds the actor to the structure.
   * 
   * @param ref		reference or label
   * @param actor	the actor to add
   */
  protected void addActor(AbstractActor actor, boolean ref) {
    if (ref)
      m_Structure.append("{" + getIndentation() + "$\\bullet$ " + toLatex(actor.getName()) + " } \n\n");
    else
      m_Details.append("{" + getIndentation() + "$\\bullet$ " + toLatex(actor.getName()) + " } \n\n");
  }
  
  /**
   * Returns the command sequence for the start of an enumerated list.
   * 
   * @return		the command sequence
   */
  protected String startEnumerate() {
    return "";
  }

  /**
   * Returns the command sequence for the end of an enumerated list.
   * 
   * @return		the command sequence
   */
  protected String endEnumerate() {
    return "";
  }
  
  /**
   * Returns the command sequence for the start of an itemized list.
   * 
   * @return		the command sequence
   */
  protected String startItemize() {
    return "";
  }

  /**
   * Returns the command sequence for the end of an itemized list.
   * 
   * @return		the command sequence
   */
  protected String endItemize() {
    return "";
  }
  
  /**
   * Makes the string LaTeX compliant.
   * 
   * @param s		the string to process
   * @return		the processed string
   */
  protected String toLatex(String s) {
    String	result;

    result = s;
    result = result.replace("%", "\\%");
    result = result.replace("$", "\\$");
    result = result.replace("_", "\\_");
    result = result.replace("{", "\\{");
    result = result.replace("}", "\\}");
    result = result.replace("@", "$@$");
    result = result.replace("^", "$\\wedge$");
    result = result.replace("*", "$*$");
    result = result.replace("\n", "\\\\ ");
    result = result.replace("\\n", "\\textbackslash n ");
    result = result.replace("\\t", "\\textbackslash t ");
    
    return result;
  }
  
  /**
   * Generates the indentation string based on the nesting level.
   * 
   * @return		the indentation string
   */
  protected String getIndentation() {
    StringBuilder	result;
    int			i;
    
    result = new StringBuilder();
    for (i = 0; i < m_Nesting.size(); i++)
      result.append("  ");
    result.append("\\hspace{" + Utils.doubleToString((0.3 * m_Nesting.size()), 1) + "cm}");
    
    return result.toString();
  }
  
  /**
   * Encapsulates the string in an item command sequence.
   * 
   * @param value	the list item
   * @return		the command sequence
   */
  protected String addItem(String value) {
    return "{" + getIndentation() + "$\\bullet$ " + toLatex(value) + " } \n\n";
  }
  
  /**
   * Encapsulates the string in an item command sequence.
   * 
   * @param name	the list item
   * @param value	the value for the name surrounded in \texttt{...}
   * @return		the command sequence
   */
  protected String addItem(String name, Object value) {
    return "{" + getIndentation() + "$\\bullet$ " + toLatex(name) + ": \\texttt{" + toLatex("" + value) + "} } \n\n";
  }
  
  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		always null
   */
  @Override
  public StringBuilder processOption(BooleanOption option) {
    Object	currValue;

    try {
      currValue = getCurrentValue(option);
      m_Details.append(addItem(getOptionIdentifier(option), currValue));
    }
    catch (Exception e) {
      System.err.println("Error obtaining current value for '" + option.getProperty() + "':");
      e.printStackTrace();
    }

    return null;
  }

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		always null
   */
  @Override
  public StringBuilder processOption(ClassOption option) {
    Object			currValue;
    Object			currValues;
    Object			value;
    int				i;
    AbstractCommandLineHandler	handler;
    boolean			isActor;

    isActor = (ClassLocator.isSubclass(AbstractActor.class, option.getBaseClass()));
    
    if (option.isVariableAttached() && !m_OutputVariableValues) {
      m_Structure.append(addItem(option.getVariable()));
      m_Details.append(addItem(getOptionIdentifier(option), option.getVariable()));
    }
    else {
      currValue  = getCurrentValue(option);
      currValues = null;

      if (currValue != null) {
	if (!option.isMultiple()) {
	  value = currValue;
	  if (isActor) {
	    m_Index++;
	    addActor((AbstractActor) value, true);
	    addActor((AbstractActor) value, false);
	  }
	  if (value instanceof OptionHandler) {
	    m_Details.append(startEnumerate());
	    m_Nesting.push(new StringBuilder());
	    doProduce(((OptionHandler) value).getOptionManager());
	    m_Nesting.pop();
	    m_Details.append(endEnumerate());
	  }
	  else {
	    handler = AbstractCommandLineHandler.getHandler(value);
	    m_Details.append(addItem(getOptionIdentifier(option), handler.toCommandLine(value)));
	  }
	}
	else {
	  currValues = currValue;
	  m_Details.append(addItem(getOptionIdentifier(option)));
	  m_Details.append(startEnumerate());
	  m_Nesting.push(new StringBuilder());
	  for (i = 0; i < Array.getLength(currValues); i++) {
	    value = Array.get(currValues, i);
	    if (isActor) {
	      m_Index++;
	      addActor((AbstractActor) value, true);
	      addActor((AbstractActor) value, false);
	    }
	    if (value instanceof OptionHandler) {
	      m_Details.append(startEnumerate());
	      m_Nesting.push(new StringBuilder());
	      doProduce(((OptionHandler) value).getOptionManager());
	      m_Nesting.pop();
	      m_Details.append(endEnumerate());
	    }
	    else {
	      handler = AbstractCommandLineHandler.getHandler(value);
	      m_Details.append(addItem(getOptionIdentifier(option), handler.toCommandLine(value)));
	    }
	  }
	  m_Nesting.pop();
	  m_Details.append(endEnumerate());
	}
      }
    }

    return null;
  }

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		always null
   */
  @Override
  public StringBuilder processOption(AbstractArgumentOption option) {
    Object	currValue;

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      m_Details.append(addItem(getOptionIdentifier(option), option.getVariable()));
    }
    else {
      currValue = getCurrentValue(option);
      if (currValue != null) {
	if (!option.isMultiple()) {
	  if (!(option instanceof AbstractNumericOption))
	    currValue = option.toString(currValue);
	  m_Details.append(addItem(getOptionIdentifier(option), currValue));
	}
	else {
	  m_Details.append(addItem(getOptionIdentifier(option), currValue));
	}
      }
    }

    return null;
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  @Override
  public String toString() {
    StringBuilder	result;
    AbstractActor	actor;

    actor = null;
    if (m_Input instanceof AbstractActor);
      actor = (AbstractActor) m_Input;

    result = new StringBuilder();
    result.append("% Generated by " + Environment.getInstance().getProject() + "\n");
    result.append("% " + new Date() + "\n");
    result.append("% " + System.getProperty("user.name") + "\n");
    result.append("\n");
    result.append("\\documentclass[a4paper," + (m_Landscape ? "landscape" : "") + "]{article}\n");
    result.append("\\usepackage{hyperref}\n");
    result.append("\\usepackage{dirtree} % http://www.tex.ac.uk/ctan/macros/generic/dirtree/\n");
    result.append("\\setlength{\\parskip}{0pt}\n");
    result.append("\\begin{document}\n");
    result.append("\\section{Structure}\n");
    if (actor != null) {
      result.append("\\textbf{" + toLatex(actor.getName()) + "}\n");
      if (!actor.getAnnotations().isEmpty()) {
	result.append("\\ ");
	result.append("\\textit{" + toLatex(actor.getAnnotations().getValue()) + "}\n\n");
      }
    }
    result.append(startItemize());
    result.append(m_Structure.toString());
    result.append(endItemize());
    result.append("\\section{Details}\n");
    if (actor != null) {
      result.append(toLatex(actor.getName()) + "\n");
    }
    result.append(startEnumerate());
    result.append(m_Details.toString());
    result.append(endEnumerate());
    result.append("\\end{document}\n");
    
    return result.toString();
  }

  /**
   * Hook-method before starting visiting options.
   */
  @Override
  protected void preProduce() {
    super.preProduce();

    m_Structure = new StringBuilder();
    m_Details   = new StringBuilder();
    m_Index     = 0;
  }

  /**
   * Returns the description of the file format.
   *
   * @return		the description
   */
  public String getFileFormat() {
    return "LaTeX";
  }

  /**
   * Returns the default file extension (without the dot).
   *
   * @return		the default extension
   */
  public String getDefaultFileExtension() {
    return "tex";
  }

  /**
   * Returns the file extensions (without the dot).
   *
   * @return		the extensions
   */
  public String[] getFileExtensions() {
    return new String[]{getDefaultFileExtension()};
  }
  
  public static void main(String[] args) throws Exception {
    Environment.setEnvironmentClass(Environment.class);
    AbstractActor actor = ActorUtils.read(args[0]);
    String latex = AbstractOptionProducer.toString(LatexProducer.class, actor);
    FileUtils.writeToFile(args[1], latex, false);
  }
}
