/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * LatexSetupPanel.java
 * Copyright (C) 2017 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.application;

import adams.core.Properties;
import adams.core.io.FileUtils;
import adams.doc.latex.LatexHelper;
import adams.env.Environment;
import adams.env.LatexDefinition;
import adams.gui.chooser.DirectoryChooserPanel;
import adams.gui.core.BaseTextField;
import adams.gui.core.ParameterPanel;

import java.awt.BorderLayout;

/**
 * Panel for configuring the system-wide LaTeX settings.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class LatexSetupPanel
  extends AbstractPreferencesPanel {

  /** for serialization. */
  private static final long serialVersionUID = -7937644706618374284L;

  /** the parameters. */
  protected ParameterPanel m_PanelParameters;

  /** the directory with the binaries. */
  protected DirectoryChooserPanel m_ChooserLatexBinaries;

  /** the executable. */
  protected BaseTextField m_TextExecutable;

  /** the executable options. */
  protected BaseTextField m_TextExecutableOptions;

  /** the bibtex executable. */
  protected BaseTextField m_TextBibtex;

  /**
   * Initializes the members.
   */
  @Override
  protected void initGUI() {
    super.initGUI();

    setLayout(new BorderLayout());

    m_PanelParameters = new ParameterPanel();
    add(m_PanelParameters, BorderLayout.CENTER);

    m_ChooserLatexBinaries = new DirectoryChooserPanel();
    m_ChooserLatexBinaries.setCurrent(LatexHelper.getBinariesDir());
    m_PanelParameters.addParameter("_Binaries dir", m_ChooserLatexBinaries);

    m_TextExecutable = new BaseTextField(10);
    m_TextExecutable.setText(LatexHelper.getExecutable());
    m_PanelParameters.addParameter("_Executable", m_TextExecutable);

    m_TextExecutableOptions = new BaseTextField(10);
    m_TextExecutableOptions.setText(LatexHelper.getExecutableOptions());
    m_PanelParameters.addParameter("Executable _options", m_TextExecutableOptions);

    m_TextBibtex = new BaseTextField(10);
    m_TextBibtex.setText(LatexHelper.getBibtex());
    m_PanelParameters.addParameter("B_ibtex", m_TextBibtex);
  }

  /**
   * Turns the parameters in the GUI into a properties object.
   *
   * @return		the properties
   */
  protected Properties toProperties() {
    Properties	result;

    result = new Properties();

    result.setProperty(LatexHelper.BINARIES_DIR, m_ChooserLatexBinaries.getCurrent().getAbsolutePath());
    result.setProperty(LatexHelper.EXECUTABLE, m_TextExecutable.getText());
    result.setProperty(LatexHelper.EXECUTABLE_OPTIONS, m_TextExecutableOptions.getText());
    result.setProperty(LatexHelper.BIBTEX, m_TextBibtex.getText());

    return result;
  }

  /**
   * The title of the preference panel.
   * 
   * @return		the title
   */
  @Override
  public String getTitle() {
    return "Latex";
  }

  /**
   * Returns whether the panel requires a wrapper scrollpane/panel for display.
   * 
   * @return		true if wrapper required
   */
  @Override
  public boolean requiresWrapper() {
    return true;
  }
  
  /**
   * Activates the twitter setup.
   * 
   * @return		null if successfully activated, otherwise error message
   */
  @Override
  public String activate() {
    boolean	result;

    result = LatexHelper.writeProperties(toProperties());
    if (result)
      return null;
    else
      return "Failed to save LaTeX setup to " + LatexHelper.FILENAME + "!";
  }

  /**
   * Returns whether the panel supports resetting the options.
   *
   * @return		true if supported
   */
  public boolean canReset() {
    String	props;

    props = Environment.getInstance().getCustomPropertiesFilename(LatexDefinition.KEY);
    return (props != null) && FileUtils.fileExists(props);
  }

  /**
   * Resets the settings to their default.
   *
   * @return		null if successfully reset, otherwise error message
   */
  public String reset() {
    String	props;

    props = Environment.getInstance().getCustomPropertiesFilename(LatexDefinition.KEY);
    if ((props != null) && FileUtils.fileExists(props)) {
      if (!FileUtils.delete(props))
	return "Failed to remove custom Latex properties: " + props;
    }

    return null;
  }
}
