/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * GPSDecimalDegrees.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.data.gps;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import adams.core.Utils;

/**
 * GPS coordinates in decimal notation.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6817 $
 */
public class GPSDecimalDegrees
  extends AbstractGPS {

  /** for serialization. */
  private static final long serialVersionUID = 1902709328711736523L;

  /** the regular expression. */
  public static final String FORMAT = "([NnSs\\+-]?)\\s*(\\d+)\\.(\\d+)\\s*([WwEe\\+-]?)\\s*(\\d+)\\.(\\d+)";

  /**
   * Default constructor.
   */
  public GPSDecimalDegrees() {
    super();
  }

  /**
   * Initialize GPS with latitude and longitude in string representation.
   * 
   * @param s		the string representation to parse
   */
  public GPSDecimalDegrees(String s) {
    super(s);
  }

  /**
   * Initialize GPS with latitude and longitude from the specified object.
   * 
   * @param gps		the GPS object to use the lat/lon from
   */
  public GPSDecimalDegrees(AbstractGPS gps) {
    super(gps);
  }

  /**
   * Initialize GPS with latitude and longitude (in decimal notation).
   * 
   * @param lat		latitude
   * @param lon		longitude
   */
  public GPSDecimalDegrees(double lat, double lon ) {
    super(lat, lon);
  }

  /**
   * Initialize GPS with latitude and longitude.
   * 
   * @param lat		latitude
   * @param lon		longitude
   */
  public GPSDecimalDegrees(Coordinate lat, Coordinate lon) {
    super(lat, lon);
  }

  /**
   * Creates a copy of itself.
   * 
   * @return		the copy
   */
  @Override
  protected GPSDecimalDegrees clone() {
    return new GPSDecimalDegrees(this);
  }

  /**
   * Parses the string to get the long/lat values from.
   * 
   * @param s		the string to parse
   */
  @Override
  public void fromString(String s) {
    s = preprocess(s);
    Pattern pattern = Pattern.compile(FORMAT);
    Matcher matcher = pattern.matcher(s);
    double latsign = 1;
    double longsign = 1;
    if (matcher.matches()) {
      String slatsign=matcher.group(1);
      String slongsign=matcher.group(4);
      if (slatsign.equalsIgnoreCase("S") || slatsign.equals("-"))
	latsign = -1;
      if (slongsign.equalsIgnoreCase("E") || slongsign.equals("-"))
	longsign = -1;
      double latdegrees  = Integer.parseInt(matcher.group(2)) * latsign;
      double longdegrees = Integer.parseInt(matcher.group(5)) * longsign;
      
      latdegrees  += Double.parseDouble("." + matcher.group(3));
      longdegrees += Double.parseDouble("." + matcher.group(6));
      
      m_Latitude  = new Coordinate(latdegrees);
      m_Longitude = new Coordinate(longdegrees);
    }
  }

  /**
   * Turns the GPS object back into its string representation.
   * 
   * @return		the string representation
   */
  @Override
  public String toString() {
    StringBuilder	result;
    double		lat;
    double		lon;
    
    result = new StringBuilder();
    lat    = m_Latitude.toDecimal();
    lon    = m_Longitude.toDecimal();
    
    if (lat < 0)
      result.append("S");
    else
      result.append("N");
    result.append(Utils.doubleToString(Math.abs(lat), NUM_DECIMALS));

    result.append(" ");

    if (lon < 0)
      result.append("E");
    else
      result.append("W");
    result.append(Utils.doubleToString(Math.abs(lon), NUM_DECIMALS));
    
    return result.toString();
  }
}
