/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * GPSDecimalSeconds.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.data.gps;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import adams.core.Utils;

/**
 * GPS coordinates with degrees and minutes as integers and seconds 
 * in decimal notation.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6817 $
 */
public class GPSDecimalSeconds
  extends AbstractGPS {

  /** for serialization. */
  private static final long serialVersionUID = 1902709328711736523L;

  /** the regular expression. */
  public static final String FORMAT = "([NnSs\\+-]?)\\s*(\\d+)\\s*(\\d+)\\s+(\\d+)\\.(\\d+)\\s*([WwEe\\+-]?)\\s*(\\d+)\\s*(\\d+)\\s+(\\d+)\\.(\\d+)\\s*";

  /**
   * Default constructor.
   */
  public GPSDecimalSeconds() {
    super();
  }

  /**
   * Initialize GPS with latitude and longitude in string representation.
   * 
   * @param s		the string representation to parse
   */
  public GPSDecimalSeconds(String s) {
    super(s);
  }

  /**
   * Initialize GPS with latitude and longitude from the specified object.
   * 
   * @param gps		the GPS object to use the lat/lon from
   */
  public GPSDecimalSeconds(AbstractGPS gps) {
    super(gps);
  }

  /**
   * Initialize GPS with latitude and longitude (in decimal notation).
   * 
   * @param lat		latitude
   * @param lon		longitude
   */
  public GPSDecimalSeconds(double lat, double lon ) {
    super(lat, lon);
  }

  /**
   * Initialize GPS with latitude and longitude.
   * 
   * @param lat		latitude
   * @param lon		longitude
   */
  public GPSDecimalSeconds(Coordinate lat, Coordinate lon) {
    super(lat, lon);
  }

  /**
   * Creates a copy of itself.
   * 
   * @return		the copy
   */
  @Override
  protected GPSDecimalSeconds clone() {
    return new GPSDecimalSeconds(this);
  }

  /**
   * Parses the string to get the long/lat values from.
   * 
   * @param s		the string to parse
   */
  @Override
  public void fromString(String s) {
    s = preprocess(s);
    Pattern pattern = Pattern.compile(FORMAT);
    Matcher matcher = pattern.matcher(s);
    double latsign = 1;
    double longsign = 1;
    if (matcher.matches()) {
      String slatsign=matcher.group(1);
      String slongsign=matcher.group(6);
      if (slatsign.equalsIgnoreCase("S") || slatsign.equals("-"))
	latsign = -1;
      if (slongsign.equalsIgnoreCase("E") || slongsign.equals("-"))
	longsign = -1;
      int latdegrees  = Integer.parseInt(matcher.group(2)) * (int)latsign;
      int longdegrees = Integer.parseInt(matcher.group(7)) * (int)longsign;
      int latminutes  = Integer.parseInt(matcher.group(3));
      int longminutes = Integer.parseInt(matcher.group(8));
      double latseconds  = Double.parseDouble(matcher.group(4) + "." + matcher.group(5));
      double longseconds = Double.parseDouble(matcher.group(9) + "." + matcher.group(10));
      
      m_Latitude  = new Coordinate(latdegrees, latminutes, latseconds);
      m_Longitude = new Coordinate(longdegrees, longminutes, longseconds);
    }
  }

  /**
   * Turns the GPS object back into its string representation.
   * 
   * @return		the string representation
   */
  @Override
  public String toString() {
    StringBuilder	result;
    double		lat;
    double		lon;
    
    result = new StringBuilder();
    lat    = m_Latitude.toDecimal();
    lon    = m_Longitude.toDecimal();
    
    if (lat < 0)
      result.append("S");
    else
      result.append("N");
    result.append(Integer.toString(Math.abs(m_Latitude.getDegree())));
    result.append(" ");
    result.append(Integer.toString(Math.abs(m_Latitude.getMinute())));
    result.append(" ");
    result.append(Utils.doubleToString(Math.abs(m_Latitude.getSecond()), NUM_DECIMALS));

    result.append(" ");
    
    if (lon < 0)
      result.append("E");
    else
      result.append("W");
    result.append(Integer.toString(Math.abs(m_Longitude.getDegree())));
    result.append(" ");
    result.append(Integer.toString(Math.abs(m_Longitude.getMinute())));
    result.append(" ");
    result.append(Utils.doubleToString(Math.abs(m_Longitude.getSecond()), NUM_DECIMALS));
    
    return result.toString();
  }
}
