/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractRandomizableFileBasedDatasetPreparation.java
 * Copyright (C) 2018 University of Waikato, Hamilton, NZ
 */

package adams.flow.transformer.preparefilebaseddataset;

import adams.core.Randomizable;
import gnu.trove.list.TIntList;
import gnu.trove.list.array.TIntArrayList;

import java.util.Random;

/**
 * Ancestor for schemes that randomize the files.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 */
public abstract class AbstractRandomizableFileBasedDatasetPreparation<T>
  extends AbstractFileBasedDatasetPreparation<T>
  implements Randomizable {

  private static final long serialVersionUID = -5423856112374249044L;

  /** the seed. */
  protected long m_Seed;

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
      "seed", "seed",
      1L);
  }

  /**
   * Sets the seed value.
   *
   * @param value	the seed
   */
  public void setSeed(long value) {
    m_Seed = value;
    reset();
  }

  /**
   * Returns the seed value.
   *
   * @return  		the seed
   */
  public long getSeed() {
    return m_Seed;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String seedTipText() {
    return "The seed value for randomizing the data.";
  }

  /**
   * Randomizes the order.
   *
   * @param data	the files to randomize
   * @return		the randomized files
   */
  protected String[] randomize(String[] data) {
    String[] 	result;
    Random 	rand;
    TIntList 	indices;
    int		i;

    rand = new Random(m_Seed);
    indices = new TIntArrayList();
    for (i = 0; i < data.length; i++)
      indices.add(i);
    indices.shuffle(rand);
    result = new String[data.length];
    for (i = 0; i < indices.size(); i++)
      result[i] = data[indices.get(i)];

    return result;
  }
}
