/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MOAStream.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import moa.options.ClassOption;
import moa.streams.InstanceStream;
import moa.streams.generators.AgrawalGenerator;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.MOAUtils;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.provenance.ProvenanceSupporter;

/**
 <!-- globalinfo-start -->
 * Generates artificial data using a MOA stream generator.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;weka.core.Instance<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MOAStream
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-stream &lt;moa.options.ClassOption&gt; (property: streamGenerator)
 * &nbsp;&nbsp;&nbsp;The stream generator to use for generating the weka.core.Instance objects.
 * &nbsp;&nbsp;&nbsp;default: moa.streams.generators.AgrawalGenerator
 * </pre>
 *
 * <pre>-num-examples &lt;int&gt; (property: numExamples)
 * &nbsp;&nbsp;&nbsp;The number of examples to generate; -1 means infinite.
 * &nbsp;&nbsp;&nbsp;default: 10000
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class MOAStream
  extends AbstractSource
  implements ProvenanceSupporter {

  /** for serialization. */
  private static final long serialVersionUID = 1862828539481494711L;

  /** the actual stream generator to use. */
  protected InstanceStream m_ActualStreamGenerator;

  /** the stream generator object. */
  protected ClassOption m_StreamGenerator;

  /** the maximum number of instance objects to generate. */
  protected int m_NumExamples;

  /** the number of examples generated so far. */
  protected int m_CountExamples;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates artificial data using a MOA stream generator.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "stream", "streamGenerator",
	    getDefaultOption());

    m_OptionManager.add(
	    "num-examples", "numExamples",
	    10000, -1, null);
  }

  /**
   * Initializes the members.
   */
  protected void reset() {
    super.reset();

    m_ActualStreamGenerator = null;
    m_CountExamples         = 0;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("evaluator");
    if (variable != null)
      result = variable;
    else
      result = getCurrentStreamGenerator().getClass().getName().replace("moa.streams.", "");

    variable = getOptionManager().getVariableForProperty("numExamples");
    if (variable != null)
      result += "/" + variable;
    else
      result += "/" + ((m_NumExamples == -1) ? "infinite" : m_NumExamples);

    return result;
  }

  /**
   * Returns the default stream generator.
   *
   * @return		the generator
   */
  protected InstanceStream getDefaultGenerator() {
    return new AgrawalGenerator();
  }

  /**
   * Returns the default class option.
   *
   * @return		the option
   */
  protected ClassOption getDefaultOption() {
    return new ClassOption(
	"stream",
	's',
	"The MOA stream generator to use from within ADAMS.",
	InstanceStream.class,
	getDefaultGenerator().getClass().getName().replace("moa.streams.", ""),
	getDefaultGenerator().getClass().getName());
  }

  /**
   * Sets the stream generator to use.
   *
   * @param value	the stream generator
   */
  public void setStreamGenerator(ClassOption value) {
    m_StreamGenerator = (ClassOption) value.copy();
    reset();
  }

  /**
   * Returns the data generator in use.
   *
   * @return		the data generator
   */
  public ClassOption getStreamGenerator() {
    return m_StreamGenerator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String streamGeneratorTipText() {
    return "The stream generator to use for generating the weka.core.Instance objects.";
  }

  /**
   * Returns the current stream generator, based on the class option.
   *
   * @return		the stream generator
   * @see		#getStreamGenerator()
   */
  protected InstanceStream getCurrentStreamGenerator() {
    return (moa.streams.InstanceStream) MOAUtils.fromOption(m_StreamGenerator);
  }

  /**
   * Sets the maximum number of examples to generate. Less or equal to 0
   * means infinite.
   *
   * @param value	the maximum number
   */
  public void setNumExamples(int value) {
    m_NumExamples = value;
    reset();
  }

  /**
   * Returns the number of examples to generate. Less or equal to 0 means
   * infinite.
   *
   * @return		the maximum number
   */
  public int getNumExamples() {
    return m_NumExamples;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String numExamplesTipText() {
    return "The number of examples to generate; -1 means infinite.";
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->weka.core.Instance.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Instance.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Instances	data;

    result = null;

    try {
      m_ActualStreamGenerator = getCurrentStreamGenerator();
      m_CountExamples         = 0;
      data                    = m_ActualStreamGenerator.getHeader();
      if (data == null)
	result = "No header obtained from stream generator!";
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result = new Token(m_ActualStreamGenerator.nextInstance());
    if (m_NumExamples > -1)
      m_CountExamples++;

    updateProvenance(result);

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    boolean	maxReached;

    maxReached = ((m_NumExamples > -1) && (m_CountExamples >= m_NumExamples));

    return (m_ActualStreamGenerator != null) && !maxReached && m_ActualStreamGenerator.hasMoreInstances();
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled())
      cont.addProvenance(new ProvenanceInformation(ActorType.DATAGENERATOR, this, ((Token) cont).getPayload().getClass()));
  }
}
