/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FTPConnection.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.standalone;

import java.util.Vector;

import org.apache.commons.net.ProtocolCommandEvent;
import org.apache.commons.net.ProtocolCommandListener;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;

import adams.core.Utils;
import adams.core.base.BasePassword;

/**
 <!-- globalinfo-start -->
 * Provides access to a FTP host.<br/>
 * If debugging is enabled, the FTP commands issued by other actors will get printed as debug output of this actor.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: FTPConnection
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-host &lt;java.lang.String&gt; (property: host)
 * &nbsp;&nbsp;&nbsp;The host (name&#47;IP address) to connect to.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-user &lt;java.lang.String&gt; (property: user)
 * &nbsp;&nbsp;&nbsp;The FTP user to use for connecting.
 * &nbsp;&nbsp;&nbsp;default: anonymous
 * </pre>
 *
 * <pre>-password &lt;adams.core.base.BasePassword&gt; (property: password)
 * &nbsp;&nbsp;&nbsp;The password of the FTP user to use for connecting.
 * &nbsp;&nbsp;&nbsp;default: {}
 * </pre>
 *
 * <pre>-passive (property: usePassiveMode)
 * &nbsp;&nbsp;&nbsp;If enabled, passive mode is used instead.
 * </pre>
 *
 * <pre>-binary (property: useBinaryMode)
 * &nbsp;&nbsp;&nbsp;If enabled, binary mode is used instead of ASCII.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class FTPConnection
  extends AbstractStandalone
  implements ProtocolCommandListener {

  /** for serialization. */
  private static final long serialVersionUID = -1959430342987913960L;

  /** the FTP host. */
  protected String m_Host;

  /** the FTP user to use. */
  protected String m_User;

  /** the FTP password to use. */
  protected BasePassword m_Password;

  /** whether to use passive mode. */
  protected boolean m_UsePassiveMode;

  /** whether to use binary file transfer mode. */
  protected boolean m_UseBinaryMode;

  /** the FTP client object. */
  protected FTPClient m_Client;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Provides access to a FTP host.\n"
      + "If debugging is enabled, the FTP commands issued by other actors will "
      + "get printed as debug output of this actor.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "host", "host",
	    "");

    m_OptionManager.add(
	    "user", "user",
	    "anonymous");

    m_OptionManager.add(
	    "password", "password",
	    new BasePassword(""));

    m_OptionManager.add(
	    "passive", "usePassiveMode",
	    false);

    m_OptionManager.add(
	    "binary", "useBinaryMode",
	    false);
  }

  /**
   * Resets the actor.
   */
  protected void reset() {
    super.reset();

    disconnect();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String		result;
    String		variable;
    Vector<String>	flags;

    variable = getOptionManager().getVariableForProperty("user");
    if (variable != null)
      result = variable;
    else
      result = m_User;

    variable = getOptionManager().getVariableForProperty("password");
    if (variable != null)
      result += ":" + variable;
    else
      result += ":" + m_Password.getValue().replaceAll(".", "*");

    result  += "@";
    variable = getOptionManager().getVariableForProperty("host");
    if (variable != null)
      result += variable;
    else
      result += m_Host;

    flags = new Vector<String>();
    if (m_UsePassiveMode)
      flags.add("passive");
    if (m_UseBinaryMode)
      flags.add("binary");
    if (flags.size() > 0)
      result += " (" + Utils.flatten(flags, ", ") + ")";

    return result;
  }

  /**
   * Sets the host to connect to.
   *
   * @param value	the host name/ip
   */
  public void setHost(String value) {
    m_Host = value;
  }

  /**
   * Returns the host to connect to.
   *
   * @return		the host name/ip
   */
  public String getHost() {
    return m_Host;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String hostTipText() {
    return "The host (name/IP address) to connect to.";
  }

  /**
   * Sets the FTP user to use.
   *
   * @param value	the user name
   */
  public void setUser(String value) {
    m_User = value;
  }

  /**
   * Returns the FTP user name to use.
   *
   * @return		the user name
   */
  public String getUser() {
    return m_User;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String userTipText() {
    return "The FTP user to use for connecting.";
  }

  /**
   * Sets the FTP password to use.
   *
   * @param value	the password
   */
  public void setPassword(BasePassword value) {
    m_Password = value;
  }

  /**
   * Returns the FTP password to use.
   *
   * @return		the password
   */
  public BasePassword getPassword() {
    return m_Password;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String passwordTipText() {
    return "The password of the FTP user to use for connecting.";
  }

  /**
   * Sets whether to use passive mode.
   *
   * @param value	if true passive mode is used
   */
  public void setUsePassiveMode(boolean value) {
    m_UsePassiveMode = value;
  }

  /**
   * Returns whether passive mode is used.
   *
   * @return		true if passive mode is used
   */
  public boolean getUsePassiveMode() {
    return m_UsePassiveMode;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String usePassiveModeTipText() {
    return "If enabled, passive mode is used instead.";
  }

  /**
   * Sets whether to use binary mode.
   *
   * @param value	if true binary mode is used
   */
  public void setUseBinaryMode(boolean value) {
    m_UseBinaryMode = value;
  }

  /**
   * Returns whether binary mode is used.
   *
   * @return		true if binary mode is used
   */
  public boolean getUseBinaryMode() {
    return m_UseBinaryMode;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String useBinaryModeTipText() {
    return "If enabled, binary mode is used instead of ASCII.";
  }

  /**
   * Returns the FTP client object.
   *
   * @return		the FTP client, null if not connected
   */
  public synchronized FTPClient getFTPClient() {
    return m_Client;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if ok, otherwise error message
   */
  protected String doExecute() {
    String	msg;
    String	result;
    int		reply;

    result = null;

    try {
      m_Client = new FTPClient();
      if (isDebugOn())
	m_Client.addProtocolCommandListener(this);
      m_Client.connect(m_Host);
      reply = m_Client.getReplyCode();
      if (!FTPReply.isPositiveCompletion(reply)) {
	result = "FTP server refused connection: " + reply;
      }
      else {
	if (!m_Client.login(m_User, m_Password.getValue())) {
	  result = "Failed to connect to '" + m_Host + "' as user '" + m_User + "'";
	}
	else {
	  if (m_UsePassiveMode)
	    m_Client.enterLocalPassiveMode();
	  if (m_UseBinaryMode)
	    m_Client.setFileType(FTPClient.BINARY_FILE_TYPE);
	}
      }
    }
    catch (Exception e) {
      msg = "Failed to connect to '" + m_Host + "' as user '" + m_User + "': ";
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
      m_Client = null;
      result   = msg + e;
    }

    return result;
  }

  /**
   * Disconnects the FTP session, if necessary.
   */
  protected void disconnect() {
    if (m_Client != null) {
      if (m_Client.isConnected()) {
	try {
	  m_Client.disconnect();
	}
	catch (Exception e) {
	  getSystemErr().println("Failed to disconnect from '" + m_Host + "':");
	  getSystemErr().printStackTrace(e);
	}
	m_Client.removeProtocolCommandListener(this);
      }
    }
    m_Client = null;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    disconnect();
    super.wrapUp();
  }

  /***
   * This method is invoked by a ProtocolCommandEvent source after
   * sending a protocol command to a server.
   *
   * @param event The ProtocolCommandEvent fired.
   */
  public void protocolCommandSent(ProtocolCommandEvent event) {
    debug("cmd sent: " + event.getCommand() + "/" + event.getReplyCode());
  }

  /***
   * This method is invoked by a ProtocolCommandEvent source after
   * receiving a reply from a server.
   *
   * @param event The ProtocolCommandEvent fired.
   */
  public void protocolReplyReceived(ProtocolCommandEvent event) {
    debug("reply received: " + event.getMessage() + "/" + event.getReplyCode());
  }
}
