/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Email.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.io.File;

import adams.core.Utils;
import adams.core.base.BaseText;
import adams.core.io.PlaceholderFile;
import adams.core.net.EmailAddress;
import adams.core.net.EmailHelper;
import adams.flow.core.NullToken;

/**
 <!-- globalinfo-start -->
 * Actor for sending emails. The (optional) attachments are taken from the input.<br/>
 * Variables in 'subject', 'body' and 'signature' are automatically replaced whenever the actor is executed.<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Email
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-recipient &lt;adams.core.net.EmailAddress&gt; [-recipient ...] (property: recipients)
 * &nbsp;&nbsp;&nbsp;The recipients to send the email to.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-subject &lt;java.lang.String&gt; (property: subject)
 * &nbsp;&nbsp;&nbsp;The subject of the email.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-body &lt;adams.core.base.BaseText&gt; (property: body)
 * &nbsp;&nbsp;&nbsp;The body of the email.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-signature &lt;adams.core.base.BaseText&gt; (property: signature)
 * &nbsp;&nbsp;&nbsp;The signature of the email, gets separated by an extra line consisting of
 * &nbsp;&nbsp;&nbsp;'--'.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class Email
  extends AbstractSink {

  /** for serialization. */
  private static final long serialVersionUID = -5959868605503747649L;

  /** the recipients. */
  protected EmailAddress[] m_Recipients;

  /** the subject. */
  protected String m_Subject;

  /** the body. */
  protected BaseText m_Body;

  /** the signature. */
  protected BaseText m_Signature;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Actor for sending emails. The (optional) attachments are taken from the input.\n"
      + "Variables in 'subject', 'body' and 'signature' are automatically replaced "
      + "whenever the actor is executed.\n"
      + (EmailHelper.isEnabled() ? "" : "Email support not enabled, check email setup!");
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "recipient", "recipients",
	    new EmailAddress[0]);

    m_OptionManager.add(
	    "subject", "subject",
	    "");

    m_OptionManager.add(
	    "body", "body",
	    new BaseText(""));

    m_OptionManager.add(
	    "signature", "signature",
	    new BaseText(Utils.unbackQuoteChars(EmailHelper.getDefaultSignature())));
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    int		i;

    if (!EmailHelper.isEnabled())
      return "No email support enabled, check email setup!";

    result = "<no recipients>";

    if ((m_Recipients != null) && (m_Recipients.length > 0)) {
      result = "";
      for (i = 0; i < m_Recipients.length; i++) {
	if (i > 0)
	  result += ", ";
	result += m_Recipients[i].getValue();
      }
    }

    return result;
  }

  /**
   * Sets the recipients.
   *
   * @param value	the recipients
   */
  public void setRecipients(EmailAddress[] value) {
    m_Recipients = value;
    reset();
  }

  /**
   * Returns the recipients.
   *
   * @return 		the recipients
   */
  public EmailAddress[] getRecipients() {
    return m_Recipients;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String recipientsTipText() {
    return "The recipients to send the email to.";
  }

  /**
   * Sets the subject.
   *
   * @param value	the subject
   */
  public void setSubject(String value) {
    m_Subject = value;
    reset();
  }

  /**
   * Returns the subject.
   *
   * @return 		the subject
   */
  public String getSubject() {
    return m_Subject;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String subjectTipText() {
    return "The subject of the email.";
  }

  /**
   * Sets the body of the email.
   *
   * @param value	the body
   */
  public void setBody(BaseText value) {
    m_Body = value;
    reset();
  }

  /**
   * Returns the body of the email.
   *
   * @return 		the body
   */
  public BaseText getBody() {
    return m_Body;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String bodyTipText() {
    return "The body of the email.";
  }

  /**
   * Sets the body of the email.
   *
   * @param value	the body
   */
  public void setSignature(BaseText value) {
    m_Signature = value;
    reset();
  }

  /**
   * Returns the body of the email.
   *
   * @return 		the body
   */
  public BaseText getSignature() {
    return m_Signature;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String signatureTipText() {
    return
        "The signature of the email, gets separated by an extra line "
      + "consisting of '" + EmailHelper.SIGNATURE_SEPARATOR + "'.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class, java.lang.String[].class, java.io.File.class, java.io.File[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class, String[].class, File.class, File[].class};
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if (!EmailHelper.isEnabled())
	result = "No email support enabled, check email setup!";
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    File[]	attachments;
    String[]	recipients;
    int		i;
    String	subject;
    String	body;
    String	signature;

    result = null;

    // get attachments (if any)
    attachments = null;
    if ((m_InputToken == null) || (m_InputToken instanceof NullToken)) {
      attachments = new File[0];
    }
    else if (m_InputToken.getPayload() instanceof File) {
      attachments = new File[]{(File) m_InputToken.getPayload()};
    }
    else if (m_InputToken.getPayload() instanceof File[]) {
      attachments = (File[]) m_InputToken.getPayload();
    }
    else if (m_InputToken.getPayload() instanceof String) {
      attachments = new File[]{new PlaceholderFile((String) m_InputToken.getPayload())};
    }
    else if (m_InputToken.getPayload() instanceof String[]) {
      attachments = new File[((String[]) m_InputToken.getPayload()).length];
      for (i = 0; i < attachments.length; i++)
	attachments[i] = new PlaceholderFile(((String[]) m_InputToken.getPayload())[i]);
    }

    // recipients
    recipients = new String[m_Recipients.length];
    for (i = 0; i < m_Recipients.length; i++)
      recipients[i] = m_Recipients[i].stringValue();

    // replace variables
    subject   = getVariables().expand(m_Subject);
    body      = getVariables().expand(m_Body.getValue());
    signature = getVariables().expand(m_Signature.getValue());
    body      = EmailHelper.combine(body, signature);

    if (isDebugOn())
      debug(
	    "#recipients: " + recipients.length
	  + ", #attachments: " + attachments.length
	  + ", subject: " + subject
	  + ", len(body): " + body.length());

    try {
      EmailHelper.sendMail(recipients, subject, body, attachments);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
