/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Email.java
 * Copyright (C) 2009-2013 University of Waikato, Hamilton, New Zealand
 */

package adams.core.net;

import java.io.File;
import java.net.MalformedURLException;

import javax.activation.DataHandler;
import javax.mail.Authenticator;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import adams.core.Properties;
import adams.core.Utils;
import adams.core.base.BasePassword;
import adams.env.EmailDefinition;
import adams.env.Environment;

/**
 * A helper class for emails.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6377 $
 */
public class EmailHelper {

  /** the name of the props file. */
  public final static String FILENAME = "Email.props";

  /** Whether Email support is enabled. */
  public final static String ENABLED = "Enabled";

  /** The SMTP server. */
  public final static String SMTP_SERVER = "SmtpServer";

  /** The SMTP port. */
  public final static String SMTP_PORT = "SmtpPort";

  /** Whether authentication is necessary. */
  public final static String SMTP_REQUIRES_AUTHENTICATION = "SmtpRequiresAuthentication";

  /** Whether STARTTLS is necessary. */
  public final static String SMTP_START_TLS = "SmtpStartTls";

  /** The user for the SMTP server. */
  public final static String SMTP_USER = "SmtpUser";

  /** The password the SMTP server. */
  public final static String SMTP_PASSWORD = "SmtpPassword";

  /** The timeout for the SMTP server. */
  public final static String SMTP_TIMEOUT = "SmtpTimeout";

  /** The default "from" email address. */
  public final static String DEFAULT_ADDRESS_FROM = "DefaultAddressFrom";

  /** The default "signature". */
  public final static String DEFAULT_SIGNATURE = "DefaultSignature";

  /** the system-wide property for the SMTP host. */
  public final static String KEY_SMTPHOST = "mail.smtp.host";

  /** the system-wide property for the SMTP port. */
  public final static String KEY_SMTPPORT = "mail.smtp.port";

  /** the system-wide property for StartTLS. */
  public final static String KEY_STARTTLS = "mail.smtp.starttls.enable";

  /** the system-wide property for SMTP Auth. */
  public final static String KEY_SMTPAUTH = "mail.smtp.auth";

  /** the system-wide property for SMTP timeout. */
  public final static String KEY_SMTPTIMEOUT = "mail.smtp.timeout";

  /** the separator between body and signature. */
  public final static String SIGNATURE_SEPARATOR = "--";

  /** the properties. */
  protected static Properties m_Properties;

  /**
   * Returns the underlying properties.
   *
   * @return		the properties
   */
  public synchronized static Properties getProperties() {
    if (m_Properties == null) {
      try {
	m_Properties = Environment.getInstance().read(EmailDefinition.KEY);
      }
      catch (Exception e) {
	m_Properties = new Properties();
      }
    }

    return m_Properties;
  }

  /**
   * Writes the specified properties to disk.
   *
   * @param props	the properties to write to disk
   * @return		true if successfully stored
   */
  public synchronized static boolean writeProperties() {
    return writeProperties(getProperties());
  }

  /**
   * Writes the specified properties to disk.
   *
   * @param props	the properties to write to disk
   * @return		true if successfully stored
   */
  public synchronized static boolean writeProperties(Properties props) {
    boolean	result;

    result = Environment.getInstance().write(EmailDefinition.KEY, props);
    // require reload
    m_Properties = null;

    return result;
  }

  /**
   * Returns whether email support has been enabled.
   *
   * @return		true if enabled
   */
  public static boolean isEnabled() {
    return getProperties().getBoolean(ENABLED, false);
  }

  /**
   * Returns the SMTP server.
   *
   * @return		the server
   */
  public static String getSmtpServer() {
    return getProperties().getString(SMTP_SERVER, "somehost");
  }

  /**
   * Returns the SMTP port.
   *
   * @return		the port
   */
  public static int getSmtpPort() {
    return getProperties().getInteger(SMTP_PORT, 25);
  }

  /**
   * Returns whether to start TLS.
   *
   * @return		true if to start TLS
   */
  public static boolean getSmtpStartTLS() {
    return getProperties().getBoolean(SMTP_START_TLS, false);
  }

  /**
   * Returns whether the server requires authentication.
   *
   * @return		true if authentication required
   */
  public static boolean getSmtpRequiresAuthentication() {
    return getProperties().getBoolean(SMTP_REQUIRES_AUTHENTICATION, false);
  }

  /**
   * Returns the timeout (in msecs) for the server.
   *
   * @return		the timeout
   */
  public static int getSmtpTimeout() {
    return getProperties().getInteger(SMTP_TIMEOUT, 30000);
  }

  /**
   * Returns the SMTP user.
   *
   * @return		the user
   */
  public static String getSmtpUser() {
    return getProperties().getString(SMTP_USER, "john.doe");
  }

  /**
   * Returns the SMTP password.
   *
   * @return		the password
   */
  public static BasePassword getSmtpPassword() {
    return new BasePassword(getProperties().getString(SMTP_PASSWORD, "password"));
  }

  /**
   * Returns the default FROM address.
   *
   * @return		the default address
   */
  public static String getDefaultFromAddress() {
    return getProperties().getString(DEFAULT_ADDRESS_FROM, "john.doe@nowhere.org");
  }

  /**
   * Returns the default signature.
   *
   * @return		the default signature (back quoted)
   * @see		Utils#backQuoteChars(String)
   */
  public static String getDefaultSignature() {
    return getProperties().getString(DEFAULT_SIGNATURE, "");
  }

  /**
   * Initializes the SMTP server.
   *
   * @return		the initialized properties
   */
  public static java.util.Properties initializeSmtpServer() {
    java.util.Properties 	result;

    result = System.getProperties();
    result.setProperty(KEY_SMTPHOST,    getSmtpServer());
    result.setProperty(KEY_SMTPPORT,    "" + getSmtpPort());
    result.setProperty(KEY_STARTTLS,    "" + getSmtpStartTLS());
    result.setProperty(KEY_SMTPAUTH,    "" + getSmtpRequiresAuthentication());
    result.setProperty(KEY_SMTPTIMEOUT, "" + getSmtpTimeout());

    return result;
  }

  /**
   * Initializes the SMTP session.
   *
   * @param props	the system properties
   * @return		the session
   */
  public static Session initializeSmtpSession(java.util.Properties props) {
    Session	result;

    if (getSmtpRequiresAuthentication()) {
      result = Session.getInstance(
	  props,
	  new Authenticator() {
	    @Override
	    protected javax.mail.PasswordAuthentication getPasswordAuthentication() {
	      return new PasswordAuthentication(getSmtpUser(), getSmtpPassword().getValue());
	    }
	  });
    }
    else {
      result = Session.getInstance(props);
    }

    return result;
  }

  /**
   * Creates a new email message.
   *
   * @param session		the server session
   * @param fromAddress		the sender, null to use default
   * @param toAddress		the recipient
   * @param subject		the subject
   * @return			the email message
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   */
  public static MimeMessage newMessage(Session session, String fromAddress, String toAddress, String subject) throws AddressException, MessagingException {
    return newMessage(session, fromAddress, new String[]{toAddress}, subject);
  }

  /**
   * Creates a new email message.
   *
   * @param session		the server session
   * @param fromAddress		the sender, null to use default
   * @param toAddress		the recipients
   * @param subject		the subject
   * @return			the email message
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   */
  public static MimeMessage newMessage(Session session, String fromAddress, String[] toAddress, String subject) throws AddressException, MessagingException {
    return newMessage(session, fromAddress, toAddress, null, null, subject);
  }

  /**
   * Creates a new email message.
   *
   * @param session		the server session
   * @param fromAddress		the sender, null to use default
   * @param toAddress		the recipients, can be null
   * @param ccAddress		the CC recipients, can be null
   * @param bccAddress		the BCC recipients, can be null
   * @param subject		the subject
   * @return			the email message
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   */
  public static MimeMessage newMessage(Session session, String fromAddress, String[] toAddress, String[] ccAddress, String[] bccAddress, String subject) throws AddressException, MessagingException {
    MimeMessage	message;
    int		i;
    boolean	noRecipient;

    message = new MimeMessage(session);
    if (fromAddress == null)
      message.setFrom(new InternetAddress(getDefaultFromAddress()));
    else
      message.setFrom(new InternetAddress(fromAddress));

    noRecipient = true;
    if (toAddress != null) {
      noRecipient = (toAddress.length == 0);
      for (i = 0; i < toAddress.length; i++)
	message.addRecipient(Message.RecipientType.TO, new InternetAddress(toAddress[i]));
    }
    if (ccAddress != null) {
      noRecipient = (ccAddress.length == 0);
      for (i = 0; i < ccAddress.length; i++)
	message.addRecipient(Message.RecipientType.CC, new InternetAddress(ccAddress[i]));
    }
    if (bccAddress != null) {
      noRecipient = (bccAddress.length == 0);
      for (i = 0; i < bccAddress.length; i++)
	message.addRecipient(Message.RecipientType.BCC, new InternetAddress(bccAddress[i]));
    }
    if (noRecipient)
      throw new MessagingException("No recipients specified!");

    message.setSubject(subject);

    return message;
  }

  /**
   * Sends an email with no attachments.
   *
   * @param fromAdress		the sender, null to use default
   * @param toAddress		the address to send the email to
   * @param subject		the subject of the email
   * @param body			the body of the message
   * @return			true if successfully sent
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   * @throws MalformedURLException	never
   */
  public static boolean sendMail(String fromAddress, String toAddress, String subject, String body) throws AddressException, MessagingException, MalformedURLException {
    return sendMail(fromAddress, toAddress, subject, body, new File[0]);
  }

  /**
   * Sends an email with no attachments.
   *
   * @param fromAdress		the sender, null to use default
   * @param toAddress		the addresses to send the email to
   * @param subject		the subject of the email
   * @param body			the body of the message
   * @return			true if successfully sent
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   * @throws MalformedURLException	never
   */
  public static boolean sendMail(String fromAddress, String[] toAddress, String subject, String body) throws AddressException, MessagingException, MalformedURLException {
    return sendMail(fromAddress, toAddress, subject, body, new File[0]);
  }

  /**
   * Sends an email.
   *
   * @param fromAdress		the sender, null to use default
   * @param toAddress		the address to send the email to
   * @param subject		the subject of the email
   * @param body			the body of the message
   * @param attachments		the list of files to attach
   * @return			true if successfully sent
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   * @throws MalformedURLException	in case of a malformed URL of an attachment
   */
  public static boolean sendMail(String fromAddress, String toAddress, String subject, String body, File[] attachments) throws AddressException, MessagingException, MalformedURLException {
    return sendMail(fromAddress, new String[]{toAddress}, subject, body, attachments);
  }

  /**
   * Sends an email.
   *
   * @param fromAdress		the sender, null to use default
   * @param toAddress		the addresses to send the email to
   * @param subject		the subject of the email
   * @param body			the body of the message
   * @param attachments		the list of files to attach
   * @return			true if successfully sent
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   * @throws MalformedURLException	in case of a malformed URL of an attachment
   */
  public static boolean sendMail(String fromAddress, String[] toAddress, String subject, String body, File[] attachments) throws AddressException, MessagingException, MalformedURLException {
    return sendMail(fromAddress, toAddress, null, null, subject, body, attachments);
  }

  /**
   * Sends an email.
   *
   * @param fromAdress		the sender, null to use default
   * @param toAddress		the addresses to send the email to, can be null
   * @param ccAddress		the CC addresses to send the email to, can be null
   * @param bccAddress		the BCC addresses to send the email to, can be null
   * @param subject		the subject of the email
   * @param body		the body of the message
   * @param attachments		the list of files to attach
   * @return			true if successfully sent
   * @throws AddressException	in case of invalid internet addresses
   * @throws MessagingException	in case of a messaging problem
   * @throws MalformedURLException	in case of a malformed URL of an attachment
   */
  public static boolean sendMail(String fromAddress, String[] toAddress, String[] ccAddress, String[] bccAddress, String subject, String body, File[] attachments) throws AddressException, MessagingException, MalformedURLException {
    java.util.Properties	props;
    Session 			session;
    MimeMessage 		message;
    BodyPart 			messageBodyPart;
    Multipart 			multipart;
    int				i;

    // Setup mail server
    props = initializeSmtpServer();

    // setup session
    session = initializeSmtpSession(props);

    // setup message
    message   = newMessage(session, fromAddress, toAddress, ccAddress, bccAddress, subject);
    multipart = new MimeMultipart();

    // body
    messageBodyPart = new MimeBodyPart();
    messageBodyPart.setText(body);
    multipart.addBodyPart(messageBodyPart);

    // attachments
    for (i = 0; i < attachments.length; i++) {
      messageBodyPart = new MimeBodyPart();
      messageBodyPart.setDataHandler(new DataHandler(attachments[i].toURI().toURL()));
      messageBodyPart.setFileName(attachments[i].getName());
      multipart.addBodyPart(messageBodyPart);
    }

    // set content
    message.setContent(multipart);

    // send
    Transport.send(message);

    return true;
  }

  /**
   * Combines body and signature, but only if the signatures is neither null
   * nor empty.
   *
   * @param body	the actual body of the email
   * @param signature	the signature to add, ignored if null or empty
   * @return		the extended body
   */
  public static String combine(String body, String signature) {
    if ((signature == null) || (signature.trim().length() == 0))
      return body;
    else
      return body + "\n" + SIGNATURE_SEPARATOR + "\n" + signature;
  }
}
