/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * EmailFileChooser.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.util.Collections;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import adams.core.ClassLister;
import adams.data.io.input.EmailFileReader;
import adams.data.io.input.PropertiesEmailFileReader;
import adams.data.io.output.EmailFileWriter;
import adams.data.io.output.PropertiesEmailFileWriter;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.goe.GenericObjectEditor;
import adams.gui.goe.GenericObjectEditorDialog;

import com.googlecode.vfsjfilechooser2.utils.VFSUtils;

/**
 * A specialized JFileChooser that lists all available file Readers and Writers
 * for emails.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 7123 $
 */
public class EmailFileChooser
  extends AbstractExtensionFileFilterFileChooser<EmailFileChooser.EmailFileExtensionFilter>
  implements FileTypeDeterminingFileChooser<EmailFileReader,EmailFileWriter> {

  /** for serialization. */
  private static final long serialVersionUID = -6341967475735162796L;

  /**
   * A custom filter class that stores the associated class along the
   * description and extensions.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 7123 $
   */
  public static class EmailFileExtensionFilter
    extends ExtensionFileFilter {

    /** for serialization. */
    private static final long serialVersionUID = -2968757331524482453L;

    /** the classname. */
    protected String m_Classname;

    /**
     * Constructs a filter that matches all files.
     *
     * @param classname		the classname this filter is for
     */
    public EmailFileExtensionFilter(String classname) {
      super();

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     */
    public EmailFileExtensionFilter(String classname, String description, String extension) {
      super(description, extension);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     */
    public EmailFileExtensionFilter(String classname, String description, String[] extensions) {
      super(description, extensions);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public EmailFileExtensionFilter(String classname, String description, String extension, boolean caseSensitive) {
      super(description, extension, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public EmailFileExtensionFilter(String classname, String description, String[] extensions, boolean caseSensitive) {
      super(description, extensions, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Returns the associated classname.
     *
     * @return		the classname
     */
    public String getClassname() {
      return m_Classname;
    }

    /**
     * Compares this image format with the specified filter for order. Returns a
     * negative integer, zero, or a positive integer as this filter is less
     * than, equal to, or greater than the specified filter.
     *
     * @param   o the filter to be compared.
     * @return  a negative integer, zero, or a positive integer as this object
     *		is less than, equal to, or greater than the specified object.
     */
    @Override
    public int compareTo(ExtensionFileFilter o) {
      int				result;
      EmailFileExtensionFilter	filter;

      result = super.compareTo(o);

      if (o instanceof EmailFileExtensionFilter) {
	filter = (EmailFileExtensionFilter) o;
	if (result == 0)
	  result = m_Classname.compareTo(filter.m_Classname);
      }

      return result;
    }
  }

  /** the file filters for the readers. */
  protected static Vector<EmailFileExtensionFilter> m_ReaderFileFilters;

  /** the file filters for the writers. */
  protected static Vector<EmailFileExtensionFilter> m_WriterFileFilters;

  /** the configure button. */
  protected JButton m_ConfigureButton;

  /** the checkbox for bringing up the GenericObjectEditor. */
  protected JCheckBox m_CheckBoxOptions;

  /** the note about the options dialog. */
  protected JLabel m_LabelOptions;

  /** the GOE for displaying the options of a reader/writer. */
  protected GenericObjectEditor m_Editor;

  /**
   * Constructs a FileChooser pointing to the user's default directory.
   */
  public EmailFileChooser() {
    super();
  }

  /**
   * Constructs a FileChooser using the given File as the path.
   *
   * @param currentDirectory	the path to start in
   */
  public EmailFileChooser(File currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Constructs a FileChooser using the given path.
   *
   * @param currentDirectory	the path to start in
   */
  public EmailFileChooser(String currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Further initializations.
   */
  @Override
  protected void initialize() {
    JPanel	panel;
    JPanel	panel2;

    super.initialize();

    m_CheckBoxOptions = new JCheckBox("Invoke options dialog");
    m_CheckBoxOptions.setMnemonic('I');
    m_LabelOptions = new JLabel("<html><br>Note:<br><br>Some file formats offer additional<br>options which can be customized<br>when invoking the options dialog.</html>");
    panel = new JPanel(new BorderLayout());
    panel.add(m_CheckBoxOptions, BorderLayout.NORTH);
    panel2 = new JPanel(new BorderLayout());
    panel2.add(m_LabelOptions, BorderLayout.NORTH);
    panel2.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panel.add(panel2, BorderLayout.CENTER);
    setAccessory(panel);

    m_Editor = new GenericObjectEditor(false);

    setAutoAppendExtension(true);
  }

  /**
   * Returns whether the filters have already been initialized.
   *
   * @return		true if the filters have been initialized
   */
  @Override
  protected boolean getFiltersInitialized() {
    return (m_ReaderFileFilters != null);
  }

  /**
   * Performs the actual initialization of the filters.
   */
  @Override
  protected void doInitializeFilters() {
    initFilters(true, ClassLister.getSingleton().getClassnames(EmailFileReader.class));
    initFilters(false, ClassLister.getSingleton().getClassnames(EmailFileWriter.class));
  }

  /**
   * initializes the SpreadSheetFileExtensionFilters.
   *
   * @param reader	if true then the reader filters are initialized
   * @param classnames	the classnames of the converters
   */
  protected static void initFilters(boolean reader, String[] classnames) {
    int					i;
    String 				classname;
    Class 				cls;
    String[] 				ext;
    String 				desc;
    Object		 		converter;
    EmailFileExtensionFilter 	filter;

    if (reader)
      m_ReaderFileFilters = new Vector<EmailFileExtensionFilter>();
    else
      m_WriterFileFilters  = new Vector<EmailFileExtensionFilter>();

    for (i = 0; i < classnames.length; i++) {
      classname = (String) classnames[i];

      // get data from converter
      try {
	cls       = Class.forName(classname);
	converter = cls.newInstance();
	if (reader) {
	  desc = ((EmailFileReader) converter).getFormatDescription();
	  ext  = ((EmailFileReader) converter).getFormatExtensions();
	}
	else {
	  desc = ((EmailFileWriter) converter).getFormatDescription();
	  ext  = ((EmailFileWriter) converter).getFormatExtensions();
	}
      }
      catch (Exception e) {
	System.err.println("Failed to setup '" + classname + "':");
	e.printStackTrace();
	cls       = null;
	converter = null;
	ext       = new String[0];
	desc      = "";
      }

      if (converter == null)
	continue;

      // reader?
      if (reader) {
	filter = new EmailFileExtensionFilter(classname, desc, ext);
	m_ReaderFileFilters.add(filter);
      }
      else {
	filter = new EmailFileExtensionFilter(classname, desc, ext);
	m_WriterFileFilters.add(filter);
      }
    }

    if (reader)
      Collections.sort(m_ReaderFileFilters);
    else
      Collections.sort(m_WriterFileFilters);
  }

  /**
   * Returns the file filters for opening files.
   *
   * @return		the file filters
   */
  @Override
  protected Vector<EmailFileExtensionFilter> getOpenFileFilters() {
    return m_ReaderFileFilters;
  }

  /**
   * Returns the file filters for writing files.
   *
   * @return		the file filters
   */
  @Override
  protected Vector<EmailFileExtensionFilter> getSaveFileFilters() {
    return m_WriterFileFilters;
  }

  /**
   * initializes the GUI.
   *
   * @param dialogType		the type of dialog to setup the GUI for
   */
  @Override
  protected void initGUI(int dialogType) {
    super.initGUI(dialogType);

    // initial setup
    if (dialogType == OPEN_DIALOG) {
      m_Editor.setClassType(EmailFileReader.class);
      m_Editor.setValue(getDefaultReader());
    }
    else {
      m_Editor.setClassType(EmailFileWriter.class);
      m_Editor.setValue(getDefaultWriter());
    }
    restoreLastFilter(dialogType);
  }

  /**
   * Returns the default reader.
   *
   * @return		the default reader
   */
  protected EmailFileReader getDefaultReader() {
    return new PropertiesEmailFileReader();
  }

  /**
   * Returns the default writer.
   *
   * @return		the default writer
   */
  protected EmailFileWriter getDefaultWriter() {
    return new PropertiesEmailFileWriter();
  }

  /**
   * Pops up an "Open File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  @Override
  public RETURN_TYPE showOpenDialog(Component parent) {
    RETURN_TYPE result = super.showOpenDialog(parent);

    if (result == RETURN_TYPE.APPROVE) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResultType();
	if (result == RETURN_TYPE.APPROVE)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * Pops up an "Save File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  @Override
  public RETURN_TYPE showSaveDialog(Component parent) {
    RETURN_TYPE result = super.showSaveDialog(parent);

    if (result == RETURN_TYPE.APPROVE) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResultType();
	if (result == RETURN_TYPE.APPROVE)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * returns the reader that was chosen by the user, can be null in case the
   * user aborted the dialog or the save dialog was shown.
   *
   * @return		the chosen reader, if any
   */
  public EmailFileReader getReader() {
    configureCurrentHandlerHook(OPEN_DIALOG);

    if (m_CurrentHandler instanceof EmailFileWriter)
      return null;
    else
      return (EmailFileReader) m_CurrentHandler;
  }

  /**
   * returns the writer that was chosen by the user, can be null in case the
   * user aborted the dialog or the open dialog was shown.
   *
   * @return		the chosen writer, if any
   */
  public EmailFileWriter getWriter() {
    configureCurrentHandlerHook(SAVE_DIALOG);

    if (m_CurrentHandler instanceof EmailFileReader)
      return null;
    else
      return (EmailFileWriter) m_CurrentHandler;
  }

  /**
   * sets the current converter according to the current filefilter.
   */
  @Override
  protected void updateCurrentHandlerHook() {
    String	classname;
    Object	newHandler;

    try {
      // determine current converter
      classname  = ((EmailFileExtensionFilter) getFileFilter()).getClassname();
      newHandler = Class.forName(classname).newInstance();

      if (m_CurrentHandler == null) {
	m_CurrentHandler = newHandler;
      }
      else {
	if (!m_CurrentHandler.getClass().equals(newHandler.getClass()))
	  m_CurrentHandler = newHandler;
      }
      setFileSelectionMode(SELECTION_MODE.FILES_ONLY);
    }
    catch (Exception e) {
      m_CurrentHandler = null;
      e.printStackTrace();
    }
  }

  /**
   * configures the current converter.
   *
   * @param dialogType		the type of dialog to configure for
   */
  @Override
  protected void configureCurrentHandlerHook(int dialogType) {
    String		classname;

    if (m_CurrentHandler == null) {
      classname = ((EmailFileExtensionFilter) getFileFilter()).getClassname();
      try {
	m_CurrentHandler = Class.forName(classname).newInstance();
      }
      catch (Exception e) {
	e.printStackTrace();
	m_CurrentHandler = null;
      }

      // none found?
      if (m_CurrentHandler == null)
	return;
    }
  }

  /**
   * Returns the reader for the specified file.
   *
   * @param file	the file to determine a reader for
   * @return		the reader, null if none found
   */
  public EmailFileReader getReaderForFile(File file) {
    EmailFileReader	result;

    result = null;

    for (EmailFileExtensionFilter filter: m_ReaderFileFilters) {
      if (filter.accept(VFSUtils.toFileObject(file))) {
	try {
	  result = (EmailFileReader) Class.forName(filter.getClassname()).newInstance();
	}
	catch (Exception e) {
	  System.err.println("Failed to instantiate reader '" + filter.getClassname() + "':");
	  e.printStackTrace();
	}
      }
    }

    return result;
  }

  /**
   * Returns the writer for the specified file.
   *
   * @param file	the file to determine a reader for
   * @return		the writer, null if none found
   */
  public EmailFileWriter getWriterForFile(File file) {
    EmailFileWriter	result;

    result = null;

    for (EmailFileExtensionFilter filter: m_WriterFileFilters) {
      if (filter.accept(VFSUtils.toFileObject(file))) {
	try {
	  result = (EmailFileWriter) Class.forName(filter.getClassname()).newInstance();
	}
	catch (Exception e) {
	  System.err.println("Failed to instantiate writer '" + filter.getClassname() + "':");
	  e.printStackTrace();
	}
      }
    }

    return result;
  }
}
