/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ODFSpreadSheetReader.java
 * Copyright (C) 2010-2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.InputStream;

import org.jopendocument.dom.ODPackage;
import org.jopendocument.dom.spreadsheet.Sheet;

import adams.core.Index;
import adams.core.Range;
import adams.core.Utils;
import adams.core.io.SpreadSheet.Row;

/**
 <!-- globalinfo-start -->
 * Reads ODF (Open Document Format) spreadsheet files.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-missing &lt;java.lang.String&gt; (property: missingValue)
 * &nbsp;&nbsp;&nbsp;The placeholder for missing values.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-sheet &lt;java.lang.String&gt; (property: sheetIndex)
 * &nbsp;&nbsp;&nbsp;The index of the sheet to load; An index is a number starting with 1; the
 * &nbsp;&nbsp;&nbsp;following placeholders can be used as well: first, second, third, last_2,
 * &nbsp;&nbsp;&nbsp; last_1, last
 * &nbsp;&nbsp;&nbsp;default: first
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 5563 $
 */
public class ODFSpreadSheetReader
  extends AbstractSpreadSheetReader {

  /** for serialization. */
  private static final long serialVersionUID = 4755872204697328246L;

  /** the sheet to load. */
  protected Index m_SheetIndex;

  /** the range of columns to force to be text. */
  protected Range m_TextColumns;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Reads ODF (Open Document Format) spreadsheet files.\n"
      + "If a row contains only empty cells, this is interpreted as the end "
      + "of the sheet.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "sheet", "sheetIndex",
	    new Index(Index.FIRST));

    m_OptionManager.add(
	    "text-columns", "textColumns",
	    "");
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "OpenDocument format";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{"ods"};
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_SheetIndex  = new Index();
    m_TextColumns = new Range();
  }

  /**
   * Sets the index of the sheet to load.
   *
   * @param value	the index (1-based)
   */
  public void setSheetIndex(Index value) {
    m_SheetIndex = value;
    reset();
  }

  /**
   * Returns the index of the sheet to load.
   *
   * @return		the index (1-based)
   */
  public Index getSheetIndex() {
    return m_SheetIndex;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String sheetIndexTipText() {
    return "The index of the sheet to load; " + m_SheetIndex.getExample();
  }

  /**
   * Sets the range of columns to treat as text.
   *
   * @param value	the range of columns
   */
  public void setTextColumns(String value) {
    m_TextColumns.setRange(value);
    reset();
  }

  /**
   * Returns the range of columns to treat as text.
   *
   * @return		the range of columns
   */
  public String getTextColumns() {
    return m_TextColumns.getRange();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String textColumnsTipText() {
    return "The range of columns to treat as text; " + m_TextColumns.getExample();
  }

  /**
   * Returns whether to read from an InputStream rather than a Reader when
   * using a file name.
   *
   * @return		true if to read from an InputStream
   */
  protected boolean getUseInputStream() {
    return true;
  }

  /**
   * Turns a numeric cell into a string. Tries to use "long" representation
   * if possible.
   *
   * @param s		the string to process
   * @return		the string representation
   */
  protected String numericToString(String s) {
    Double	dbl;
    long	lng;

    dbl = Double.parseDouble(s);
    lng = dbl.longValue();
    if (dbl == lng)
      return "" + lng;
    else
      return "" + dbl;
  }

  /**
   * Reads the spreadsheet content from the specified file.
   *
   * @param in		the input stream to read from
   * @return		the spreadsheet or null in case of an error
   */
  protected SpreadSheet doRead(InputStream in) {
    SpreadSheet		result;
    org.jopendocument.dom.spreadsheet.SpreadSheet	spreadsheet;
    Sheet		sheet;
    int			i;
    int			n;
    Row			row;
    Object[]		data;
    boolean		empty;
    String		cellStr;
    boolean		numeric;

    result = new adams.core.io.SpreadSheet();

    try {
      spreadsheet = org.jopendocument.dom.spreadsheet.SpreadSheet.get(new ODPackage(in));
      m_SheetIndex.setMax(spreadsheet.getSheetCount());
      sheet       = spreadsheet.getSheet(m_SheetIndex.getIntIndex());
      result.setName(sheet.getName());

      // header
      row = result.getHeaderRow();
      for (i = 0; i < sheet.getColumnCount(); i++) {
	cellStr = sheet.getCellAt(i, 0).getTextValue();
	if (cellStr.length() == 0)
	  break;
	row.addCell("" + (i+1)).setContent(cellStr);
      }

      // data
      m_TextColumns.setMax(result.getColumnCount());
      for (n = 1; n < sheet.getRowCount(); n++) {
	data  = new Object[result.getColumnCount()];
	empty = true;
	for (i = 0; i < result.getColumnCount(); i++) {
	  numeric = !m_TextColumns.isInRange(i);
	  cellStr = sheet.getCellAt(i, n).getTextValue();
	  if (Utils.isLong(cellStr)) {
	    if (numeric)
	      data[i] = Long.parseLong(cellStr);
	    else
	      data[i] = numericToString(cellStr);
	    empty = false;
	  }
	  else if (Utils.isDouble(cellStr)) {
	    if (numeric)
	      data[i] = Double.parseDouble(cellStr);
	    else
	      data[i] = numericToString(cellStr);
	    empty = false;
	  }
	  else {
	    data[i] = cellStr;
	    if (((String) data[i]).length() > 0)
	      empty = false;
	  }
	  if (data[i].equals(m_MissingValue))
	    data[i] = null;
	}
	// no more data?
	if (empty)
	  break;
	// add row
	row = result.addRow("" + result.getRowCount());
	for (i = 0; i < data.length; i++) {
	  if (data[i] == null)
	    row.addCell("" + (i+1)).setContent(SpreadSheet.MISSING_VALUE);
	  else if (data[i] instanceof Double)
	    row.addCell("" + (i+1)).setContent((Double) data[i]);
	  else if (data[i] instanceof Long)
	    row.addCell("" + (i+1)).setContent(data[i].toString(), true);
	  else
	    row.addCell("" + (i+1)).setContent((String) data[i], false);
	}
      }
    }
    catch (Exception ioe) {
      getSystemErr().println("Failed to read data:");
      getSystemErr().printStackTrace(ioe);
      result = null;
    }

    return result;
  }
}
