/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * RandomNumberExpression.java
 * Copyright (C) 2012-2014 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.condition.bool;

import java.util.HashMap;

import adams.core.QuickInfoHelper;
import adams.data.random.JavaRandomInt;
import adams.data.random.RandomNumberGenerator;
import adams.flow.core.Actor;
import adams.flow.core.Token;
import adams.parser.BooleanExpressionText;

/**
 <!-- globalinfo-start -->
 * Evaluates to 'true' if the expression evaluates to 'true'.<br>
 * The 'X' in the expression is the number generated by the random number generator.<br>
 * <br>
 * The following grammar is used for evaluating the boolean expressions:<br>
 * <br>
 * expr_list ::= '=' expr_list expr_part | expr_part ;<br>
 * expr_part ::=  expr ;<br>
 * <br>
 * expr      ::=   ( expr )<br>
 * <br>
 * # data types<br>
 *               | number<br>
 *               | string<br>
 *               | boolean<br>
 *               | date<br>
 * <br>
 * # constants<br>
 *               | true<br>
 *               | false<br>
 *               | pi<br>
 *               | e<br>
 *               | now()<br>
 *               | today()<br>
 * <br>
 * # negating numeric value<br>
 *               | -expr<br>
 * <br>
 * # comparisons<br>
 *               | expr &lt; expr<br>
 *               | expr &lt;= expr<br>
 *               | expr &gt; expr<br>
 *               | expr &gt;= expr<br>
 *               | expr = expr<br>
 *               | expr != expr (or: expr &lt;&gt; expr)<br>
 * <br>
 * # boolean operations<br>
 *               | ! expr (or: not expr)<br>
 *               | expr &amp; expr (or: expr and expr)<br>
 *               | expr | expr (or: expr or expr)<br>
 *               | if[else] ( expr , expr (if true) , expr (if false) )<br>
 *               | ifmissing ( variable , expr (default value if variable is missing) )<br>
 * <br>
 * # arithmetics<br>
 *               | expr + expr<br>
 *               | expr - expr<br>
 *               | expr * expr<br>
 *               | expr &#47; expr<br>
 *               | expr ^ expr (power of)<br>
 *               | expr % expr (modulo)<br>
 *               ;<br>
 * <br>
 * # numeric functions<br>
 *               | abs ( expr )<br>
 *               | sqrt ( expr )<br>
 *               | log ( expr )<br>
 *               | exp ( expr )<br>
 *               | sin ( expr )<br>
 *               | cos ( expr )<br>
 *               | tan ( expr )<br>
 *               | rint ( expr )<br>
 *               | floor ( expr )<br>
 *               | pow[er] ( expr , expr )<br>
 *               | ceil ( expr )<br>
 *               | year ( expr )<br>
 *               | month ( expr )<br>
 *               | day ( expr )<br>
 *               | hour ( expr )<br>
 *               | minute ( expr )<br>
 *               | second ( expr )<br>
 *               | weekday ( expr )<br>
 *               | weeknum ( expr )<br>
 * <br>
 * # string functions<br>
 *               | substr ( expr , start [, end] )<br>
 *               | left ( expr , len )<br>
 *               | mid ( expr , start , len )<br>
 *               | right ( expr , len )<br>
 *               | rept ( expr , count )<br>
 *               | concatenate ( expr1 , expr2 [, expr3-5] )<br>
 *               | lower[case] ( expr )<br>
 *               | upper[case] ( expr )<br>
 *               | trim ( expr )<br>
 *               | matches ( expr , regexp )<br>
 *               | trim ( expr )<br>
 *               | len[gth] ( str )<br>
 *               | find ( search , expr [, pos] )<br>
 *               | replace ( str , pos , len , newstr )<br>
 *               | substitute ( str , find , replace [, occurrences] )<br>
 *               ;<br>
 * <br>
 * Notes:<br>
 * - Variables are either all upper case letters (e.g., "ABC") or any character   apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").<br>
 * - 'start' and 'end' for function 'substr' are indices that start at 1.<br>
 * - Index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)<br>
 * - Line comments start with '#'<br>
 * - Semi-colons (';') or commas (',') can be used as separator in the formulas,<br>
 *   e.g., 'pow(2,2)' is equivalent to 'pow(2;2)'<br>
 * - dates have to be of format 'yyyy-MM-dd' or 'yyyy-MM-dd HH:mm:ss'<br>
 * - times have to be of format 'HH:mm:ss' or 'yyyy-MM-dd HH:mm:ss'<br>
 * - the characters in square brackets in function names are optional:<br>
 *   e.g. 'len("abc")' is the same as 'length("abc")'<br>
 * <br>
 * A lot of the functions have been modeled after LibreOffice:<br>
 *   https:&#47;&#47;help.libreoffice.org&#47;Calc&#47;Functions_by_Category<br>
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <br><br>
 * 
 * <pre>-logging-level &lt;OFF|SEVERE|WARNING|INFO|CONFIG|FINE|FINER|FINEST&gt; (property: loggingLevel)
 * &nbsp;&nbsp;&nbsp;The logging level for outputting errors and debugging output.
 * &nbsp;&nbsp;&nbsp;default: WARNING
 * </pre>
 * 
 * <pre>-expression &lt;adams.parser.BooleanExpressionText&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The expression to evaluate using the current random number obtained from 
 * &nbsp;&nbsp;&nbsp;the generator; expressions that consists solely of a variable (eg '&#64;{blah
 * &nbsp;&nbsp;&nbsp;}') get automatically wrapped in parentheses, since the expression string 
 * &nbsp;&nbsp;&nbsp;gets interpreted as attached variable instead.
 * &nbsp;&nbsp;&nbsp;default: X &lt; 500
 * </pre>
 * 
 * <pre>-generator &lt;adams.data.random.RandomNumberGenerator&gt; (property: generator)
 * &nbsp;&nbsp;&nbsp;The generator that generates a new random number each time a token arrives.
 * &nbsp;&nbsp;&nbsp;default: adams.data.random.JavaRandomInt
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 10824 $
 */
public class RandomNumberExpression
  extends AbstractExpression {

  /** for serialization. */
  private static final long serialVersionUID = 5195367806715940844L;
  
  /** the random number generator. */
  protected RandomNumberGenerator m_Generator;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return
        "Evaluates to 'true' if the expression evaluates to 'true'.\n"
      + "The 'X' in the expression is the number generated by the random number generator.\n\n"
      + "The following grammar is used for evaluating the boolean expressions:\n\n"
      + getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "generator", "generator",
	    new JavaRandomInt());
  }

  /**
   * Returns the quick info string to be displayed in the flow editor.
   *
   * @return		the info or null if no info to be displayed
   */
  @Override
  public String getQuickInfo() {
    return super.getQuickInfo() + QuickInfoHelper.toString(this, "generator", m_Generator, ", generator: ");
  }

  /**
   * Returns the default expression to use.
   * 
   * @return		the default
   */
  @Override
  protected BooleanExpressionText getDefaultExpression() {
    return new BooleanExpressionText("X < 500");
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  @Override
  public String expressionTipText() {
    return
        "The expression to evaluate using the current random number obtained "
      + "from the generator; expressions that consists solely of a "
      + "variable (eg '@{blah}') get automatically wrapped in parentheses, "
      + "since the expression string gets interpreted as attached variable "
      + "instead.";
  }

  /**
   * Sets the random number generator to use.
   *
   * @param value	the generator
   */
  public void setGenerator(RandomNumberGenerator value) {
    m_Generator = value;
    reset();
  }

  /**
   * Returns the random number generator in use.
   *
   * @return		the generator
   */
  public RandomNumberGenerator getGenerator() {
    return m_Generator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String generatorTipText() {
    return "The generator that generates a new random number each time a token arrives.";
  }

  /**
   * Evaluates the expression.
   *
   * @param owner	the owning actor
   * @param token	the current token passing through
   * @return		true if the expression evaluates to 'true'
   */
  @Override
  protected boolean doEvaluate(Actor owner, Token token) {
    String	exp;
    HashMap	symbols;

    exp     = owner.getVariables().expand(getExpression().getValue());
    symbols = new HashMap();
    symbols.put("X", m_Generator.next().doubleValue());

    return doEvaluate(exp, symbols);
  }
}
