/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractSpreadSheetColumnConverter.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.data.conversion;

import adams.core.Index;
import adams.data.spreadsheet.Cell;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;

/**
 * Ancestor for column converter schemes.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6291 $
 * @param <T> the type of class to convert the cell values to
 */
public abstract class AbstractSpreadSheetColumnConverter<T>
  extends AbstractInPlaceSpreadSheetConversion {

  /** for serialization. */
  private static final long serialVersionUID = 8681800940519018023L;

  /** the column to process. */
  protected Index m_Column;

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "column", "column",
	    new Index(Index.FIRST));
  }

  /**
   * Sets the column to convert.
   *
   * @param value	the index
   */
  public void setColumn(Index value) {
    m_Column = value;
    reset();
  }

  /**
   * Returns the column to convert.
   *
   * @return		the index
   */
  public Index getColumn() {
    return m_Column;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public abstract String columnTipText();

  /**
   * Prepares the conversion.
   * 
   * @param input	the spreadsheet to convert
   */
  protected void preConvert(SpreadSheet input) {
    m_Column.setMax(input.getColumnCount());
  }
  
  /**
   * Converts the cell's content to a new format.
   * 
   * @param cellOld	the current cell
   * @param cellNew	the new cell with the converted content
   * @throws Exception	if conversion fails
   */
  protected abstract void convert(Cell cellOld, Cell cellNew) throws Exception;
  
  /**
   * Generates the new spreadsheet from the input.
   * 
   * @param input	the incoming spreadsheet
   * @return		the generated spreadsheet
   * @throws Exception	if conversion fails for some reason
   */
  @Override
  protected SpreadSheet convert(SpreadSheet input) throws Exception {
    SpreadSheet	result;
    int		i;
    int		r;
    int		index;
    Row		newRow;
    Row		row;
    Cell	cell;
    Cell	newCell;
    
    preConvert(input);
    
    if (m_NoCopy)
      result = input;
    else
      result = input.getHeader();
    index = m_Column.getIntIndex();
    
    for (r = 0; r < input.getRowCount(); r++) {
      row = input.getRow(r);
      if (m_NoCopy)
	newRow = row;
      else
	newRow = result.addRow();
      for (i = 0; i < input.getColumnCount(); i++) {
	cell = row.getCell(i);
	if (cell == null)
	  continue;
	newCell = newRow.addCell(i);
	if (cell.isMissing()) {
	  newCell.setMissing();
	  continue;
	}
	if (i == index) {
	  try {
	    convert(cell, newCell);
	  }
	  catch (Exception e) {
	    getSystemErr().println("Failed to convert value '" + cell.getContent() + "' at " + SpreadSheet.getCellPosition(r, i));
	    newCell.setMissing();
	  }
	}
	else {
	  newCell.assign(cell);
	}
      }
    }
    
    return result;
  }
}
