/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * MergeSpreadSheetColumns.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.data.conversion;

import java.util.HashSet;

import adams.core.Range;
import adams.data.spreadsheet.DataRow;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;

/**
 <!-- globalinfo-start -->
 * Merges two or more columns in a spreadsheet into a single column.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-columns &lt;adams.core.Range&gt; (property: columns)
 * &nbsp;&nbsp;&nbsp;The range of columns to merge into a single one; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: first-last
 * </pre>
 * 
 * <pre>-glue &lt;java.lang.String&gt; (property: glue)
 * &nbsp;&nbsp;&nbsp;The 'glue' string to use between two values that get merged.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6301 $
 */
public class MergeSpreadSheetColumns
  extends AbstractSpreadSheetConversion {

  /** for serialization. */
  private static final long serialVersionUID = -5364554292793461868L;

  /** the columns to merge. */
  protected Range m_Columns;
  
  /** the concatenation string to use. */
  protected String m_Glue;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Merges two or more columns in a spreadsheet into a single column.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "columns",
	    new Range(Range.ALL));

    m_OptionManager.add(
	    "glue", "glue",
	    "");
  }

  /**
   * Sets the range of columns to merge.
   *
   * @param value	the range
   */
  public void setColumns(Range value) {
    m_Columns = value;
    reset();
  }

  /**
   * Returns the range of columns to merge.
   *
   * @return		true range
   */
  public Range getColumns() {
    return m_Columns;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String columnsTipText() {
    return "The range of columns to merge into a single one; " + m_Columns.getExample();
  }

  /**
   * Sets the "glue" to use between two columns.
   *
   * @param value	the glue
   */
  public void setGlue(String value) {
    m_Glue = value;
    reset();
  }

  /**
   * Returns the "glue" to use between two columns.
   *
   * @return		the glue
   */
  public String getGlue() {
    return m_Glue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String glueTipText() {
    return "The 'glue' string to use between two values that get merged.";
  }

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  protected SpreadSheet convert(SpreadSheet input) throws Exception {
    SpreadSheet		result;
    int[]		indicesMerge;
    HashSet<Integer>	hashMerge;
    String		headerMerged;
    int			i;
    Row			headerInput;
    Row			headerOutput;
    Row			rowOutput;
    int			n;
    String		content;
    int			indexMerged;
    boolean		first;
    
    m_Columns.setMax(input.getColumnCount());
    indicesMerge = m_Columns.getIntIndices();
    if (indicesMerge.length < 2) {
      getSystemErr().println("Need at least two column indices to merge!");
      return input;
    }

    hashMerge = new HashSet<Integer>();
    for (int index: indicesMerge)
      hashMerge.add(index);
    
    result = input.getHeader();
    result.getHeaderRow().clear();
    
    // header
    indexMerged  = -1;
    headerMerged = "";
    for (i = 0; i < indicesMerge.length; i++) {
      if (i > 0)
	headerMerged += m_Glue;
      headerMerged += input.getHeaderRow().getCell(indicesMerge[i]);
    }
    headerInput  = input.getHeaderRow();
    headerOutput = result.getHeaderRow();
    n            = 0;
    first        = true;
    for (i = 0; i < input.getColumnCount(); i++) {
      if (hashMerge.contains(i)) {
	if (headerMerged != null) {
	  indexMerged  = n;
	  headerOutput.addCell("" + n).setContentAsString(headerMerged);
	  headerMerged = null;
	}
	if (first) {
	  first = false;
	  n++;
	}
      }
      else {
	headerOutput.addCell("" + n).setContent(headerInput.getCell(i).getContent());
	n++;
      }
    }
    
    // data
    for (DataRow rowInput: input.rows()) {
      rowOutput = result.addRow();
      first     = true;
      n         = 0;
      for (i = 0; i < input.getColumnCount(); i++) {
	if (hashMerge.contains(i)) {
	  content = SpreadSheet.MISSING_VALUE;
	  if (rowInput.hasCell(i) && !rowInput.getCell(i).isMissing())
	    content = rowInput.getCell(i).getContent();
	  if (!rowOutput.hasCell(indexMerged) || rowOutput.getCell(indexMerged).isMissing())
	    rowOutput.addCell("" + n).setContentAsString(content);
	  else
	    rowOutput.getCell(indexMerged).setContentAsString(rowOutput.getCell(indexMerged).getContent() + m_Glue + content);
	  if (first) {
	    first = false;
	    n++;
	  }
	}
	else {
	  if (rowInput.hasCell(i) && !rowInput.getCell(i).isMissing())
	    rowOutput.addCell("" + n).setContent(rowInput.getCell(i).getContent());
	  n++;
	}
      }
    }
  
    return result;
  }
}
