/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetDbWriter.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.HashSet;

import adams.data.spreadsheet.Cell;
import adams.data.spreadsheet.Cell.ContentType;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;
import adams.db.SQL;
import adams.flow.core.ActorUtils;

/**
 <!-- globalinfo-start -->
 * Transfers a SpreadSheet object into a database.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetDbWriter
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-table &lt;java.lang.String&gt; (property: table)
 * &nbsp;&nbsp;&nbsp;The table to write the data to (gets automatically created).
 * &nbsp;&nbsp;&nbsp;default: blah
 * </pre>
 * 
 * <pre>-column-name-conversion &lt;AS_IS|LOWER_CASE|UPPER_CASE&gt; (property: columnNameConversion)
 * &nbsp;&nbsp;&nbsp;How to convert the column headers into SQL table column names.
 * &nbsp;&nbsp;&nbsp;default: UPPER_CASE
 * </pre>
 * 
 * <pre>-max-string-length &lt;int&gt; (property: maxStringLength)
 * &nbsp;&nbsp;&nbsp;The maximum length for strings to enforce; can be used as &#64;MAX in the 'stringColumnsSQL'
 * &nbsp;&nbsp;&nbsp; property.
 * &nbsp;&nbsp;&nbsp;default: 50
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-string-column-sql &lt;java.lang.String&gt; (property: stringColumnSQL)
 * &nbsp;&nbsp;&nbsp;The SQL type to use for STRING columns in the CREATE statement; you can 
 * &nbsp;&nbsp;&nbsp;use the &#64;MAX placeholder to tie the type to the 'naxStringLength' property;
 * &nbsp;&nbsp;&nbsp; see also: http:&#47;&#47;en.wikipedia.org&#47;wiki&#47;SQL
 * &nbsp;&nbsp;&nbsp;default: VARCHAR(&#64;MAX)
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6332 $
 */
public class SpreadSheetDbWriter
  extends AbstractSink {

  /** for serialization. */
  private static final long serialVersionUID = 393925191813730213L;

  /**
   * Defines how column names are converted.
   * 
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 6332 $
   */
  public enum ColumnNameConversion {
    /** no change. */
    AS_IS,
    /** lower case. */
    LOWER_CASE,
    /** upper case. */
    UPPER_CASE
  }
  
  /** the placeholder for the maximum length for string values. */
  public final static String PLACEHOLDER_MAX = "@MAX";
  
  /** the database connection. */
  protected adams.db.AbstractDatabaseConnection m_DatabaseConnection;

  /** the table to write the data to. */
  protected String m_Table;

  /** the type used for the table. */
  protected ContentType[] m_Types;

  /** the maximum length for column names. */
  protected int m_MaxColumnLength;
  
  /** the column names (shortened, disambiguated). */
  protected String[] m_ColumnNames;
  
  /** the column name conversion. */
  protected ColumnNameConversion m_ColumnNameConversion;
  
  /** the SQL type for string columns. */
  protected String m_StringColumnSQL;
  
  /** the maximum length for strings. */
  protected int m_MaxStringLength;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Transfers a SpreadSheet object into a database.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "table", "table",
	    "blah");

    m_OptionManager.add(
	    "column-name-conversion", "columnNameConversion",
	    ColumnNameConversion.UPPER_CASE);

    m_OptionManager.add(
	    "max-string-length", "maxStringLength",
	    50, 1, null);

    m_OptionManager.add(
	    "string-column-sql", "stringColumnSQL",
	    "VARCHAR(" +  PLACEHOLDER_MAX + ")");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "table: ";
    variable = getOptionManager().getVariableForProperty("table");
    if (variable != null)
      result += variable;
    else
      result += m_Table;

    result += ", conversion: ";
    variable = getOptionManager().getVariableForProperty("columnNameConversion");
    if (variable != null)
      result += variable;
    else
      result += m_ColumnNameConversion;

    result += ", max string: ";
    variable = getOptionManager().getVariableForProperty("maxStringLength");
    if (variable != null)
      result += variable;
    else
      result += m_MaxStringLength;

    result += ", string type: ";
    variable = getOptionManager().getVariableForProperty("stringColumnSQL");
    if (variable != null)
      result += variable;
    else
      result += m_StringColumnSQL;
    
    return result;
  }

  /**
   * Sets the table to write the data to.
   *
   * @param value	the table name
   */
  public void setTable(String value) {
    m_Table = value;
    reset();
  }

  /**
   * Returns the table to write the data to.
   *
   * @return		the table name
   */
  public String getTable() {
    return m_Table;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String tableTipText() {
    return "The table to write the data to (gets automatically created).";
  }

  /**
   * Sets how to convert the column headers into SQL table column names.
   *
   * @param value	the conversion
   */
  public void setColumnNameConversion(ColumnNameConversion value) {
    m_ColumnNameConversion = value;
    reset();
  }

  /**
   * Returns how to convert the column headers into SQL table column names.
   *
   * @return		the conversion
   */
  public ColumnNameConversion getColumnNameConversion() {
    return m_ColumnNameConversion;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String columnNameConversionTipText() {
    return "How to convert the column headers into SQL table column names.";
  }

  /**
   * Sets the maximum length for strings.
   *
   * @param value	the maximum
   */
  public void setMaxStringLength(int value) {
    m_MaxStringLength = value;
    reset();
  }

  /**
   * Returns the maximum length for strings.
   *
   * @return		the maximum
   */
  public int getMaxStringLength() {
    return m_MaxStringLength;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String maxStringLengthTipText() {
    return 
	"The maximum length for strings to enforce; can be used "
	+ "as " + PLACEHOLDER_MAX + " in the 'stringColumnsSQL' property.";
  }

  /**
   * Sets the SQL type for string columns for the CREATE statement.
   *
   * @param value	the SQL type
   */
  public void setStringColumnSQL(String value) {
    m_StringColumnSQL = value;
    reset();
  }

  /**
   * Returns the SQL type for string columns for the CREATE statement.
   *
   * @return		the SQL type
   */
  public String getStringColumnSQL() {
    return m_StringColumnSQL;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String stringColumnSQLTipText() {
    return 
	"The SQL type to use for STRING columns in the CREATE statement; "
	+ "you can use the " + PLACEHOLDER_MAX + " placeholder to tie the type "
	+ "to the 'naxStringLength' property; see also: http://en.wikipedia.org/wiki/SQL";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.spreadsheet.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Determines the database connection in the flow.
   *
   * @return		the database connection to use
   */
  protected adams.db.AbstractDatabaseConnection getDatabaseConnection() {
    return ActorUtils.getDatabaseConnection(
	  this,
	  adams.flow.standalone.DatabaseConnection.class,
	  adams.db.DatabaseConnection.getSingleton());
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      m_DatabaseConnection = getDatabaseConnection();

    return result;
  }
  
  /**
   * Fixes the column name.
   * 
   * @param s		the column name to fix
   * @return		the fixed name
   */
  protected String fixColumnName(String s) {
    String	result;
    int		i;
    char	chr;
    
    result = "";
    
    for (i = 0; i < s.length(); i++) {
      chr = s.charAt(i);
      if ((chr >= 'A') && (chr <= 'Z'))
	result += chr;
      else if ((chr >= 'a') && (chr <= 'z'))
	result += chr;
      if (i >= 0) {
	if ((chr >= '0') && (chr <= '9'))
	  result += chr;
	else if (chr == '_')
	  result += chr;
      }
    }
    
    // too long?
    if (result.length() > m_MaxColumnLength)
      result = result.substring(0, m_MaxColumnLength);

    // convert name
    switch (m_ColumnNameConversion) {
      case AS_IS:
	// nothing
	break;
      case LOWER_CASE:
	result = result.toLowerCase();
	break;
      case UPPER_CASE:
	result = result.toUpperCase();
	break;
      default:
	throw new IllegalStateException("Unhandled conversion type: " + m_ColumnNameConversion);
    }
    
    return result;
  }
  
  /**
   * Determines the column types and names.
   * 
   * @param sheet	the sheet to analyze
   * @see		#m_Types
   * @see		#m_ColumnNames
   */
  protected String determineSetup(SpreadSheet sheet) {
    String		result;
    int			i;
    ContentType		type;
    HashSet<String>	names;
    String		name;
    String		prefix;
    int			count;
    DatabaseMetaData	meta;

    result = null;
    
    // maximum length of column names
    m_MaxColumnLength = -1;
    try {
      meta = m_DatabaseConnection.getConnection(false).getMetaData();
      m_MaxColumnLength = meta.getMaxColumnNameLength();
      if (m_MaxColumnLength == 0)
	m_MaxColumnLength = Integer.MAX_VALUE;
    }
    catch (Exception e) {
      result = handleException("Failed to obtain database meta-data!", e);
    }

    // column types
    m_Types = new ContentType[sheet.getColumnCount()];
    if (result == null) {
      for (i = 0; i < sheet.getColumnCount(); i++) {
	m_Types[i] = ContentType.STRING;
	if (sheet.isNumeric(i)) {
	  type = sheet.getContentType(i);
	  if (type == ContentType.LONG)
	    m_Types[i] = ContentType.LONG;
	  else if (type == ContentType.DOUBLE)
	    m_Types[i] = ContentType.DOUBLE;
	}
	else {
	  type = sheet.getContentType(i);
	  if (type == null)
	    type = ContentType.STRING;
	  switch (type) {
	    case TIME:
	    case DATE:
	      m_Types[i] = type;
	      break;
	  }
	}
      }
    }

    // column names
    m_ColumnNames = new String[sheet.getColumnCount()];
    if (result == null) {
      names = new HashSet<String>();
      for (i = 0; i < sheet.getColumnCount(); i++) {
	name   = sheet.getHeaderRow().getCell(i).getContent();
	name   = fixColumnName(name);
	prefix = name;
	count  = 0;
	while (names.contains(name)) {
	  count++;
	  if (new String(prefix + count).length() > m_MaxColumnLength)
	    prefix = prefix.substring(0, prefix.length() - 1);
	  name = prefix + count;
	}
	names.add(name);
	m_ColumnNames[i] = name;
      }      
    }
    
    if (isDebugOn()) {
      for (i = 0; i < sheet.getColumnCount(); i++)
	debug(sheet.getHeaderRow().getCell(i).getContent() + ": " + m_ColumnNames[i] + ", " + m_Types[i]);
    }
    
    return result;
  }
  
  /**
   * Creates the table.
   * 
   * @param sql		for executing queries
   * @param sheet	the data to write to the database
   * @return		null if everything OK, otherwise error message
   */
  protected String createTable(SQL sql, SpreadSheet sheet) {
    String		result;
    StringBuilder	query;
    int			i;
    Boolean		rs;
    String		stringType;

    result = null;

    stringType = m_StringColumnSQL.replace(PLACEHOLDER_MAX, "" + m_MaxStringLength);
    query      = new StringBuilder("CREATE TABLE " + m_Table + "(");
    for (i = 0; i < sheet.getColumnCount(); i++) {
      if (i > 0)
	query.append(", ");
      switch (m_Types[i]) {
	case LONG:
	  query.append(m_ColumnNames[i] + " INTEGER");
	  break;
	case DOUBLE:
	  query.append(m_ColumnNames[i] + " DOUBLE PRECISION");
	  break;
	case DATE:
	  query.append(m_ColumnNames[i] + " TIMESTAMP");
	  break;
	case TIME:
	  query.append(m_ColumnNames[i] + " TIME");
	  break;
	default:
	  query.append(m_ColumnNames[i] + " " + stringType);
	  break;
      }
    }
    query.append(");");
    try {
      if (isDebugOn())
	debug("Creating table: " + query);
      rs = sql.execute(query.toString());
      if (rs == null)
	result = "Failed to create table, check console!";
    }
    catch (Exception e) {
      result = handleException("Failed to create table '" + m_Table + "' using: " + query, e);
    }
    
    return result;
  }

  /**
   * Writes the data to the table.
   * 
   * @param sql		for performing the writing
   * @param sheet	the data to write
   * @return		null if everything OK, otherwise error message
   */
  protected String writeData(SQL sql, SpreadSheet sheet) {
    String		result;
    StringBuilder	query;
    PreparedStatement	stmt;
    int			i;
    Cell		cell;
    int			type;
    int			count;
    String		str;
    
    result = null;
    
    query = new StringBuilder("INSERT INTO " + m_Table + "(");
    for (i = 0; i < sheet.getColumnCount(); i++) {
      if (i > 0)
	query.append(", ");
      query.append(m_ColumnNames[i]);
    }
    query.append(") VALUES (");
    for (i = 0; i < sheet.getColumnCount(); i++) {
      if (i > 0)
	query.append(", ");
      query.append("?");
    }
    query.append(");");
    
    try {
      stmt = sql.prepareStatement(query.toString());
    }
    catch (Exception e) {
      result = handleException("Failed to prepare statement: " + query, e);
      stmt   = null;
    }

    if (result == null) {
      count = 0;
      for (Row row: sheet.rows()) {
	if (m_Stopped)
	  break;
	count++;
	try {
	  for (i = 0; i < sheet.getColumnCount(); i++) {
	    cell = row.getCell(i);
	    if ((cell == null) || cell.isMissing()) {
	      switch (m_Types[i]) {
		case DATE:
		  type = Types.TIMESTAMP;
		  break;
		case TIME:
		  type = Types.TIME;
		  break;
		case DOUBLE:
		  type = Types.DOUBLE;
		  break;
		case LONG:
		  type = Types.INTEGER;
		  break;
		default:
		  type = Types.VARCHAR;
		  break;
	      }
	      stmt.setNull(i + 1, type);
	    }
	    else {
	      switch (m_Types[i]) {
		case DATE:
		  stmt.setDate(i + 1, new java.sql.Date(cell.toDate().getTime()));
		  break;
		case TIME:
		  stmt.setTime(i + 1, cell.toTime());
		  break;
		case DOUBLE:
		  stmt.setDouble(i + 1, cell.toDouble());
		  break;
		case LONG:
		  stmt.setInt(i + 1, cell.toLong().intValue());
		  break;
		default:
		  str = cell.getContent();
		  if (str.length() > m_MaxStringLength)
		    str = str.substring(0, m_MaxStringLength);
		  stmt.setString(i + 1, str);
		  break;
	      }
	    }
	  }
	  stmt.execute();
	}
	catch (Exception e) {
	  result = handleException("Failed to insert data: " + row + "\nusing: " + stmt, e);
	  break;
	}
	if (count % 1000 == 0)
	  getSystemOut().println(count + " rows processed");
      }
    }
    
    SQL.close(stmt);
    
    return result;
  }
  
  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String		result;
    SpreadSheet		sheet;
    SQL			sql;

    result = null;

    sheet   = (SpreadSheet) m_InputToken.getPayload();
    sql     = new SQL(m_DatabaseConnection);

    // determine column types
    result = determineSetup(sheet);
    
    // create table?
    if ((result == null) && !sql.tableExists(m_Table))
      result = createTable(sql, sheet);
    
    // write data
    if (result == null)
      result = writeData(sql, sheet);

    return result;
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  @Override
  public void wrapUp() {
    m_DatabaseConnection = null;

    super.wrapUp();
  }
}
