/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetDisplay.java
 * Copyright (C) 2009-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;
import java.awt.Color;
import java.util.Arrays;
import java.util.Date;
import java.util.Vector;

import javax.swing.JTable;

import adams.core.DateUtils;
import adams.data.spreadsheet.SpreadSheet;
import adams.flow.core.Token;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.ColorHelper;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.SpreadSheetTable;
import adams.gui.core.SpreadSheetTableModel;
import adams.gui.sendto.SendToActionUtils;

/**
 <!-- globalinfo-start -->
 * Actor for displaying a spreadsheet.<br/>
 * Custom background for negative&#47;positive values can be specified as well.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetDisplay
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-short-title (property: shortTitle)
 * &nbsp;&nbsp;&nbsp;If enabled uses just the name for the title instead of the actor's full 
 * &nbsp;&nbsp;&nbsp;name.
 * </pre>
 * 
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 640
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 480
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 * 
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 * 
 * <pre>-font &lt;java.awt.Font&gt; (property: font)
 * &nbsp;&nbsp;&nbsp;The font of the dialog.
 * &nbsp;&nbsp;&nbsp;default: Monospaced-PLAIN-12
 * </pre>
 * 
 * <pre>-num-decimals &lt;int&gt; (property: numDecimals)
 * &nbsp;&nbsp;&nbsp;The number of decimals for numeric values.
 * &nbsp;&nbsp;&nbsp;default: 3
 * &nbsp;&nbsp;&nbsp;minimum: -1
 * </pre>
 * 
 * <pre>-use-custom-negative-background (property: useCustomNegativeBackground)
 * &nbsp;&nbsp;&nbsp;Whether to use a custom background color for negative values.
 * </pre>
 * 
 * <pre>-negative-background &lt;java.awt.Color&gt; (property: negativeBackground)
 * &nbsp;&nbsp;&nbsp;The custom background for negative values (must be enabled).
 * &nbsp;&nbsp;&nbsp;default: #ffffff
 * </pre>
 * 
 * <pre>-use-custom-positive-background (property: useCustomPositiveBackground)
 * &nbsp;&nbsp;&nbsp;Whether to use a custom background color for positive values.
 * </pre>
 * 
 * <pre>-positive-background &lt;java.awt.Color&gt; (property: positiveBackground)
 * &nbsp;&nbsp;&nbsp;The custom background for positive values (must be enabled).
 * &nbsp;&nbsp;&nbsp;default: #ffffff
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6228 $
 */
public class SpreadSheetDisplay
  extends AbstractTextualDisplay
  implements DisplayPanelProvider {

  /** for serialization. */
  private static final long serialVersionUID = 3247255046513744115L;

  /** the table. */
  protected SpreadSheetTable m_Table;

  /** the table model. */
  protected SpreadSheetTableModel m_TableModel;

  /** the number of decimals for numeric values. */
  protected int m_NumDecimals;
  
  /** whether to use custom background color for negative values. */
  protected boolean m_UseCustomNegativeBackground;
  
  /** the custom background color for negative values. */
  protected Color m_NegativeBackground;
  
  /** whether to use custom background color for positive values. */
  protected boolean m_UseCustomPositiveBackground;
  
  /** the custom background color for positive values. */
  protected Color m_PositiveBackground;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return 
	"Actor for displaying a spreadsheet.\n"
	+ "Custom background for negative/positive values can be specified as well.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "num-decimals", "numDecimals",
	    3, -1, null);

    m_OptionManager.add(
	    "use-custom-negative-background", "useCustomNegativeBackground",
	    false);

    m_OptionManager.add(
	    "negative-background", "negativeBackground",
	    Color.WHITE);

    m_OptionManager.add(
	    "use-custom-positive-background", "useCustomPositiveBackground",
	    false);

    m_OptionManager.add(
	    "positive-background", "positiveBackground",
	    Color.WHITE);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = super.getQuickInfo();
    
    variable = getOptionManager().getVariableForProperty("numDecimals");
    result  += ", decimals: ";
    if (variable != null)
      result += variable;
    else
      result += m_NumDecimals;
    
    if (m_UseCustomNegativeBackground) {
      result += ", negative: ";
      variable = getOptionManager().getVariableForProperty("negativeBackground");
      if (variable != null)
        result += variable;
      else
        result += ColorHelper.toHex(m_NegativeBackground);
    }
    
    if (m_UseCustomPositiveBackground) {
      result += ", positive: ";
      variable = getOptionManager().getVariableForProperty("positiveBackground");
      if (variable != null)
        result += variable;
      else
        result += ColorHelper.toHex(m_PositiveBackground);
    }
    
    return result;
  }

  /**
   * Sets the number of decimals to display.
   *
   * @param value 	the number of decimals
   */
  public void setNumDecimals(int value) {
    m_NumDecimals = value;
    reset();
  }

  /**
   * Returns the currently set number of decimals to display.
   *
   * @return 		the number of decimals
   */
  public int getNumDecimals() {
    return m_NumDecimals;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String numDecimalsTipText() {
    return "The number of decimals for numeric values.";
  }

  /**
   * Sets whether to use a custom background for negative values.
   *
   * @param value 	true if to use custom color
   */
  public void setUseCustomNegativeBackground(boolean value) {
    m_UseCustomNegativeBackground = value;
    reset();
  }

  /**
   * Returns whether to use a custom background for negative values.
   *
   * @return 		true if custom color
   */
  public boolean getUseCustomNegativeBackground() {
    return m_UseCustomNegativeBackground;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String useCustomNegativeBackgroundTipText() {
    return "Whether to use a custom background color for negative values.";
  }

  /**
   * Sets the custom background color for negative values.
   *
   * @param value 	the color
   */
  public void setNegativeBackground(Color value) {
    m_NegativeBackground = value;
    reset();
  }

  /**
   * Returns the custom background color for negative values.
   *
   * @return 		the number of decimals
   */
  public Color getNegativeBackground() {
    return m_NegativeBackground;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String negativeBackgroundTipText() {
    return "The custom background for negative values (must be enabled).";
  }

  /**
   * Sets whether to use a custom background for positive values.
   *
   * @param value 	true if to use custom color
   */
  public void setUseCustomPositiveBackground(boolean value) {
    m_UseCustomPositiveBackground = value;
    reset();
  }

  /**
   * Returns whether to use a custom background for positive values.
   *
   * @return 		true if custom color
   */
  public boolean getUseCustomPositiveBackground() {
    return m_UseCustomPositiveBackground;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String useCustomPositiveBackgroundTipText() {
    return "Whether to use a custom background color for positive values.";
  }

  /**
   * Sets the custom background color for positive values.
   *
   * @param value 	the color
   */
  public void setPositiveBackground(Color value) {
    m_PositiveBackground = value;
    reset();
  }

  /**
   * Returns the custom background color for positive values.
   *
   * @return 		the number of decimals
   */
  public Color getPositiveBackground() {
    return m_PositiveBackground;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String positiveBackgroundTipText() {
    return "The custom background for positive values (must be enabled).";
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  @Override
  protected int getDefaultWidth() {
    return 640;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  @Override
  protected int getDefaultHeight() {
    return 480;
  }

  /**
   * Clears the content of the panel.
   */
  @Override
  public void clearPanel() {
    m_TableModel = new SpreadSheetTableModel(new SpreadSheet());
    m_Table      = new SpreadSheetTable(m_TableModel);
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  @Override
  public BasePanel newPanel() {
    BasePanel	result;

    result       = new BasePanel(new BorderLayout());
    m_TableModel = new SpreadSheetTableModel(new SpreadSheet());
    m_Table      = new SpreadSheetTable(m_TableModel);
    result.add(new BaseScrollPane(m_Table), BorderLayout.CENTER);

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.spreadsheet.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  @Override
  protected void display(Token token) {
    m_TableModel = new SpreadSheetTableModel((SpreadSheet) token.getPayload());
    m_Table.setModel(m_TableModel);
    m_Table.setNumDecimals(m_NumDecimals);
    if (m_UseCustomNegativeBackground)
      m_Table.setNegativeBackground(m_NegativeBackground);
    else
      m_Table.setNegativeBackground(null);
    if (m_UseCustomPositiveBackground)
      m_Table.setPositiveBackground(m_PositiveBackground);
    else
      m_Table.setPositiveBackground(null);
  }

  /**
   * Creates a new panel for the token.
   *
   * @param token	the token to display in a new panel
   * @return		the generated panel
   */
  public AbstractDisplayPanel createDisplayPanel(Token token) {
    AbstractDisplayPanel	result;

    result = new AbstractTextDisplayPanel("SpreadSheet") {
      private static final long serialVersionUID = 3524967045456783678L;
      protected SpreadSheetTable m_Table;
      protected SpreadSheetTableModel m_TableModel;
      @Override
      protected void initGUI() {
	super.initGUI();
	setLayout(new BorderLayout());
	m_TableModel = new SpreadSheetTableModel(new SpreadSheet());
	m_Table      = new SpreadSheetTable(m_TableModel);
	add(new BaseScrollPane(m_Table), BorderLayout.CENTER);
      }
      @Override
      public void display(Token token) {
	m_TableModel = new SpreadSheetTableModel((SpreadSheet) token.getPayload());
	m_Table.setModel(m_TableModel);
	m_Table.setNumDecimals(m_NumDecimals);
	if (m_UseCustomNegativeBackground)
	  m_Table.setNegativeBackground(m_NegativeBackground);
	else
	  m_Table.setNegativeBackground(null);
	if (m_UseCustomPositiveBackground)
	  m_Table.setPositiveBackground(m_PositiveBackground);
	else
	  m_Table.setPositiveBackground(null);
      }
      @Override
      public ExtensionFileFilter getCustomTextFileFilter() {
	return ExtensionFileFilter.getCsvFileFilter();
      }
      @Override
      public String supplyText() {
	return m_TableModel.getSheet().toString();
      }
      @Override
      public void clearPanel() {
	m_TableModel = new SpreadSheetTableModel();
	m_Table.setModel(m_TableModel);
      }
      public void cleanUp() {
      }
    };
    result.display(token);

    return result;
  }

  /**
   * Returns whether the created display panel requires a scroll pane or not.
   *
   * @return		true if the display panel requires a scroll pane
   */
  public boolean displayPanelRequiresScrollPane() {
    return false;
  }

  /**
   * Executes the flow item. 
   * <p/>
   * Outputs the token on the command-line in headless mode.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String	result;
    
    result = null;
    
    if (m_Headless) {
      getSystemOut().println("\n--> " + DateUtils.getTimestampFormatterMsecs().format(new Date()) + "\n");
      getSystemOut().println(m_InputToken.getPayload());
    }
    else {
      result = super.doExecute();
    }
    
    return result;
  }

  /**
   * Returns a custom file filter for the file chooser.
   * 
   * @return		the file filter, null if to use default one
   */
  @Override
  public ExtensionFileFilter getCustomTextFileFilter() {
    return ExtensionFileFilter.getCsvFileFilter();
  }

  /**
   * Returns the text to save.
   *
   * @return		the text, null if no text available
   */
  @Override
  public String supplyText() {
    String	result;

    result = null;

    if (m_TableModel != null)
      result = m_TableModel.getSheet().toString();

    return result;
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  @Override
  public Class[] getSendToClasses() {
    Vector<Class> 	result;
    
    result = new Vector<Class>(Arrays.asList(super.getSendToClasses()));
    if (!result.contains(JTable.class))
      result.add(JTable.class);
    
    return result.toArray(new Class[result.size()]);
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  @Override
  public Object getSendToItem(Class[] cls) {
    Object	result;

    result = null;

    if (SendToActionUtils.isAvailable(JTable.class, cls)) {
      result = m_Table;
      if (m_Table.getRowCount() == 0)
	result = null;
    }
    else {
      result = super.getSendToItem(cls);
    }

    return result;
  }
}
