/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetDbReader.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.source;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Types;

import adams.core.Utils;
import adams.data.spreadsheet.DataRowType;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;
import adams.db.SQL;
import adams.db.SQLStatement;
import adams.flow.core.ActorUtils;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Returns a spreadsheet object generated from an SQL query.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetDbReader
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-query &lt;adams.db.SQLStatement&gt; (property: query)
 * &nbsp;&nbsp;&nbsp;The SQL query to use for generating the spreadsheet.
 * &nbsp;&nbsp;&nbsp;default: select * from blah
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6293 $
 */
public class SpreadSheetDbReader
  extends AbstractSource {

  /** for serialization. */
  private static final long serialVersionUID = 494594301273926225L;

  /** the SQL query to execute. */
  protected SQLStatement m_Query;

  /** the data row type to use. */
  protected DataRowType m_DataRowType;
  
  /** the generated output token. */
  protected Token m_OutputToken;

  /** the database connection. */
  protected adams.db.AbstractDatabaseConnection m_DatabaseConnection;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Returns a spreadsheet object generated from an SQL query.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "query", "query",
	    new SQLStatement("select * from blah"));

    m_OptionManager.add(
	    "data-row-type", "dataRowType",
	    DataRowType.DENSE);
  }

  /**
   * Resets the scheme.
   */
  @Override
  protected void reset() {
    super.reset();
    
    m_OutputToken = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;
    String	variable;

    result   = "query: ";
    variable = getOptionManager().getVariableForProperty("query");
    if (variable != null)
      result += variable;
    else
      result += Utils.shorten(m_Query.stringValue(), 40);

    result  += ",row type: ";
    variable = getOptionManager().getVariableForProperty("dataRowType");
    if (variable != null)
      result += variable;
    else
      result += m_DataRowType;

    return result;
  }

  /**
   * Sets the comma-separated list of column names.
   *
   * @param value	the list
   */
  public void setQuery(SQLStatement value) {
    m_Query = value;
    reset();
  }

  /**
   * Returns the comma-separated list of column names.
   *
   * @return		the list
   */
  public SQLStatement getQuery() {
    return m_Query;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String queryTipText() {
    return "The SQL query to use for generating the spreadsheet.";
  }

  /**
   * Sets the type of data row to use.
   *
   * @param value	the type
   */
  public void setDataRowType(DataRowType value) {
    m_DataRowType = value;
    reset();
  }

  /**
   * Returns the type of data row to use.
   *
   * @return		the type
   */
  public DataRowType getDataRowType() {
    return m_DataRowType;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String dataRowTypeTipText() {
    return "The type of row to use for the data.";
  }

  /**
   * Determines the database connection in the flow.
   *
   * @return		the database connection to use
   */
  protected adams.db.AbstractDatabaseConnection getDatabaseConnection() {
    return ActorUtils.getDatabaseConnection(
	  this,
	  adams.flow.standalone.DatabaseConnection.class,
	  adams.db.DatabaseConnection.getSingleton());
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      m_DatabaseConnection = getDatabaseConnection();

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the Class of the generated tokens
   */
  @Override
  public Class[] generates() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String		result;
    SpreadSheet		sheet;
    SQL			sql;
    ResultSet		rs;
    Row			row;
    ResultSetMetaData	meta;
    int			i;
    int[]		type;
    
    result = null;

    sheet = null;
    rs    = null;
    try {
      sheet = new SpreadSheet();
      sheet.setDataRowClass(m_DataRowType.getRowClass());
      sql   = new SQL(m_DatabaseConnection);
      rs    = sql.getResultSet(m_Query.getValue());

      // header
      row  = sheet.getHeaderRow();
      meta = rs.getMetaData();
      type = new int[meta.getColumnCount()];
      for (i = 1; i <= meta.getColumnCount(); i++) {
	row.addCell("" + i).setContentAsString(meta.getColumnName(i));
	type[i - 1] = meta.getColumnType(i);
      }

      // data
      while (rs.next() && !m_Stopped) {
	row = sheet.addRow();
	for (i = 1; i <= sheet.getColumnCount(); i++) {
	  switch (type[i - 1]) {
	    case Types.TIME:
	      row.addCell(i - 1).setContent(rs.getTime(i));
	      break;
	    case Types.DATE:
	      row.addCell(i - 1).setContent(rs.getDate(i));
	      break;
	    case Types.TIMESTAMP:
	      row.addCell(i - 1).setContent(rs.getTimestamp(i));
	      break;
	    case Types.INTEGER:
	      row.addCell(i - 1).setContent(rs.getInt(i));
	      break;
	    case Types.FLOAT:
	      row.addCell(i - 1).setContent(rs.getFloat(i));
	      break;
	    case Types.DOUBLE:
	      row.addCell(i - 1).setContent(rs.getDouble(i));
	      break;
	    default:
	      row.addCell(i - 1).setContent(rs.getString(i));
	      break;
	  }
	  if (rs.wasNull())
	    row.getCell(i - 1).setMissing();
	}
      }
    }
    catch (Exception e) {
      result = handleException("Failed to retrieve data from database", e);
    }
    
    SQL.closeAll(rs);
    
    if (m_Stopped)
      result = null;
    
    if (result == null)
      m_OutputToken = new Token(sheet);
    
    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  @Override
  public Token output() {
    Token	result;
    
    result        = m_OutputToken;
    m_OutputToken = null;
    
    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  @Override
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  @Override
  public void wrapUp() {
    m_DatabaseConnection = null;

    super.wrapUp();
  }
}
