/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetInfo.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import adams.core.Index;
import adams.data.spreadsheet.Cell.ContentType;
import adams.data.spreadsheet.SpreadSheet;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Outputs statistics of a SpreadSheet object.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetInfo
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-type &lt;NUM_COLUMNS|NUM_ROWS|COLUMN_NAME|COLUMN_TYPE|CELL_TYPES&gt; (property: type)
 * &nbsp;&nbsp;&nbsp;The type of information to generate.
 * &nbsp;&nbsp;&nbsp;default: NUM_ROWS
 * </pre>
 * 
 * <pre>-column-index &lt;adams.core.Index&gt; (property: columnIndex)
 * &nbsp;&nbsp;&nbsp;The column index to use for generating column-specific information. An index 
 * &nbsp;&nbsp;&nbsp;is a number starting with 1; the following placeholders can be used as well:
 * &nbsp;&nbsp;&nbsp; first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: last
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6286 $
 */
public class SpreadSheetInfo
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -3019442578354930841L;

  /**
   * The type of information to generate.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 6286 $
   */
  public enum InfoType {
    /** the number of columns. */
    NUM_COLUMNS,
    /** the number of rows. */
    NUM_ROWS,
    /** the name of the column (at specified index). */
    COLUMN_NAME,
    /** the overall column type (at specified index). */
    COLUMN_TYPE,
    /** all cell types (at specified index). */
    CELL_TYPES
  }

  /** the type of information to generate. */
  protected InfoType m_Type;

  /** the index of the column to get the information for. */
  protected Index m_ColumnIndex;

  /** the output queue. */
  protected List m_Queue;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Outputs statistics of a SpreadSheet object.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "type", "type",
	    InfoType.NUM_ROWS);

    m_OptionManager.add(
	    "column-index", "columnIndex",
	    new Index(Index.LAST));
  }

  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_ColumnIndex = new Index();
    m_Queue       = new ArrayList();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String		result;
    String		variable;
    HashSet<InfoType>	types;

    result = null;

    variable = getOptionManager().getVariableForProperty("type");
    if (variable != null)
      result = variable;
    else
      result = "" + m_Type;

    types = new HashSet<InfoType>(
	Arrays.asList(
	    new InfoType[]{
		InfoType.NUM_COLUMNS,
		InfoType.NUM_ROWS,
	    }));
    if (!types.contains(m_Type)) {
      result += ", index: ";
      variable = getOptionManager().getVariableForProperty("columnIndex");
      if (variable != null)
        result += variable;
      else
        result += m_ColumnIndex.getIndex();
    }

    return result;
  }

  /**
   * Sets the type of information to generate.
   *
   * @param value	the type
   */
  public void setType(InfoType value) {
    m_Type = value;
    reset();
  }

  /**
   * Returns the type of information to generate.
   *
   * @return		the type
   */
  public InfoType getType() {
    return m_Type;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String typeTipText() {
    return "The type of information to generate.";
  }

  /**
   * Sets the column index to use for column specific information.
   *
   * @param value	the 1-based index
   */
  public void setColumnIndex(Index value) {
    m_ColumnIndex = value;
    reset();
  }

  /**
   * Returns the column index to use for column specific information.
   *
   * @return		the 1-based index
   */
  public Index getColumnIndex() {
    return m_ColumnIndex;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String columnIndexTipText() {
    return "The column index to use for generating column-specific information. " + m_ColumnIndex.getExample();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.spreadsheet.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Integer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    switch (m_Type) {
      case COLUMN_NAME:
      case COLUMN_TYPE:
      case CELL_TYPES:
	return new Class[]{String.class};

      case NUM_COLUMNS:
      case NUM_ROWS:
	return new Class[]{Integer.class};

      default:
	throw new IllegalStateException("Unhandled info type: " + m_Type);
    }
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String			result;
    SpreadSheet			sheet;
    int				index;
    ContentType			type;
    Collection<ContentType>	types;

    result = null;

    m_Queue = new ArrayList();
    sheet   = (SpreadSheet) m_InputToken.getPayload();
    m_ColumnIndex.setMax(sheet.getColumnCount());

    switch (m_Type) {
      case COLUMN_NAME:
	index = m_ColumnIndex.getIntIndex();
	if (index != -1)
	  m_Queue.add(sheet.getHeaderRow().getCell(index).getContent());
	break;

      case NUM_COLUMNS:
	m_Queue.add(sheet.getColumnCount());
	break;

      case NUM_ROWS:
	m_Queue.add(sheet.getRowCount());
	break;
	
      case COLUMN_TYPE:
	index = m_ColumnIndex.getIntIndex();
	if (index != -1) {
	  type = sheet.getContentType(index);
	  if (type == null)
	    type = ContentType.STRING;
	  m_Queue.add(type.toString());
	}
	break;
	
      case CELL_TYPES:
	index = m_ColumnIndex.getIntIndex();
	if (index != -1) {
	  types = sheet.getContentTypes(index);
	  for (ContentType ct: types)
	    m_Queue.add(ct.toString());
	}
	break;

      default:
	result = "Unhandled info type: " + m_Type;
    }

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  @Override
  public boolean hasPendingOutput() {
    return (m_Queue != null) && (m_Queue.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  @Override
  public Token output() {
    Token	result;

    result       = new Token(m_Queue.get(0));
    m_InputToken = null;
    m_Queue.remove(0);

    return result;
  }
  
  /**
   * Cleans up after the execution has finished.
   */
  @Override
  public void wrapUp() {
    m_Queue = null;
    
    super.wrapUp();
  }
}
