/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetPlotGenerator.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Hashtable;
import java.util.Vector;

import adams.core.Index;
import adams.core.Range;
import adams.data.spreadsheet.Cell;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;
import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Outputs plot containers generated from a spreadsheet.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.container.SequencePlotterContainer<br/>
 * <p/>
 * Container information:<br/>
 * - adams.flow.container.SequencePlotterContainer: PlotName, X, Y, IsMarker
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetPlotGenerator
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-columns &lt;java.lang.String&gt; (property: plotColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to include in the plot; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-x-column &lt;java.lang.String&gt; (property: XColumn)
 * &nbsp;&nbsp;&nbsp;The (optional) index of the column which values to use as X values in the 
 * &nbsp;&nbsp;&nbsp;plot; An index is a number starting with 1; the following placeholders can 
 * &nbsp;&nbsp;&nbsp;be used as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-default-cell-value &lt;double&gt; (property: defaultCellValue)
 * &nbsp;&nbsp;&nbsp;The default value for missing or non-numeric cells.
 * &nbsp;&nbsp;&nbsp;default: -1.0
 * </pre>
 * 
 * <pre>-prefix-columns &lt;adams.core.Range&gt; (property: prefixColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to prefix the plot name with; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6084 $
 */
public class SpreadSheetPlotGenerator
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 1147935218531182101L;

  /** the default string value for cells that are missing. */
  public final static String MISSING_CELL_VALUE = "MISSING";

  /** the columns to plot. */
  protected Range m_PlotColumns;

  /** the (optional) column to use as X value. */
  protected Index m_XColumn;

  /** the default value for missing or non-numeric cells. */
  protected double m_DefaultCellValue;
  
  /** the columns to prefix the plot name with. */
  protected Range m_PrefixColumns;

  /** the generated plot containers. */
  protected Vector<SequencePlotterContainer> m_Containers;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Outputs plot containers generated from a spreadsheet.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "plotColumns",
	    "");

    m_OptionManager.add(
	    "x-column", "XColumn",
	    "");

    m_OptionManager.add(
	    "default-cell-value", "defaultCellValue",
	    -1.0);

    m_OptionManager.add(
	    "prefix-columns", "prefixColumns",
	    new Range());
  }

  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_Containers  = new Vector<SequencePlotterContainer>();
    m_PlotColumns = new Range();
    m_XColumn     = new Index();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("plotColumns");
    if (variable != null)
      return variable;
    else if (m_PlotColumns.getRange().length() > 0)
      return m_PlotColumns.getRange();
    else
      return null;
  }

  /**
   * Sets the column range to use in the plot.
   *
   * @param value	the column range
   */
  public void setPlotColumns(String value) {
    m_PlotColumns.setRange(value);
    reset();
  }

  /**
   * Returns the current column range to use in the plot.
   *
   * @return		the column range
   */
  public String getPlotColumns() {
    return m_PlotColumns.getRange();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String plotColumnsTipText() {
    return "The range of columns to include in the plot; " + m_PlotColumns.getExample();
  }

  /**
   * Sets the index of the column which values to use as X values.
   *
   * @param value	the column index
   */
  public void setXColumn(String value) {
    m_XColumn.setIndex(value);
    reset();
  }

  /**
   * Returns the index of the column which values to use as X values.
   *
   * @return		the column index
   */
  public String getXColumn() {
    return m_XColumn.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String XColumnTipText() {
    return "The (optional) index of the column which values to use as X values in the plot; " + m_XColumn.getExample();
  }

  /**
   * Sets the default value for missing or non-numeric cells.
   *
   * @param value	the default value
   */
  public void setDefaultCellValue(double value) {
    m_DefaultCellValue = value;
    reset();
  }

  /**
   * Returns the default value for missing or non-numeric cells.
   *
   * @return		the default value
   */
  public double getDefaultCellValue() {
    return m_DefaultCellValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String defaultCellValueTipText() {
    return "The default value for missing or non-numeric cells.";
  }

  /**
   * Sets the column range to generate the prefix of the plot with.
   *
   * @param value	the column range
   */
  public void setPrefixColumns(Range value) {
    m_PrefixColumns = value;
    reset();
  }

  /**
   * Returns the current column range to generate the prefix of the plot with.
   *
   * @return		the column range
   */
  public Range getPrefixColumns() {
    return m_PrefixColumns;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String prefixColumnsTipText() {
    return "The range of columns to prefix the plot name with; " + m_PlotColumns.getExample();
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.spreadsheet.SpreadSheet.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{SpreadSheet.class};
  }

  /**
   * Returns the string value for the specified cell.
   *
   * @param row		the row to get the cell from
   * @param index	the column index
   * @return		the string value
   */
  protected String getCellString(Row row, int index) {
    String	result;
    Cell	cell;

    result = MISSING_CELL_VALUE;

    cell = row.getCell(index);
    if ((cell != null) && !cell.isMissing())
      result = cell.getContent();

    return result;
  }

  /**
   * Returns the cell value for the specified column index.
   * Uses the default value if the cell is missing or non-numeric.
   *
   * @param row		the row to get the cell from
   * @param index	the column index
   * @return		the cell value
   * @see		#m_DefaultCellValue
   */
  protected double getCellValue(Row row, int index) {
    double	result;
    Cell	cell;

    result = m_DefaultCellValue;

    cell = row.getCell(index);
    if ((cell != null) && !cell.isMissing() && cell.isNumeric())
      result = Double.parseDouble(cell.getContent());

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String			result;
    SequencePlotterContainer	cont;
    SpreadSheet			sheet;
    int				i;
    int				n;
    Row				row;
    int[]			plotCols;
    String[]			plotNames;
    int				xIndex;
    int[]			prefixCols;
    String			prefix;
    Hashtable<String,Integer>	xIndexString;

    result = null;

    sheet = (SpreadSheet) m_InputToken.getPayload();
    m_PlotColumns.setMax(sheet.getColumnCount());
    m_XColumn.setMax(sheet.getColumnCount());
    m_PrefixColumns.setMax(sheet.getColumnCount());

    // gather data
    xIndex       = m_XColumn.getIntIndex();
    xIndexString = null;
    if (xIndex != -1) {
      if (!sheet.isNumeric(xIndex))
	xIndexString = new Hashtable<String,Integer>();
    }

    // create plot containers
    m_Containers.clear();
    plotCols   = m_PlotColumns.getIntIndices();
    plotNames  = new String[plotCols.length];
    prefixCols = m_PrefixColumns.getIntIndices();
    for (i = 0; i < plotNames.length; i++)
      plotNames[i] = sheet.getHeaderRow().getCell(plotCols[i]).getContent();
    for (i = 0; i < sheet.getRowCount(); i++) {
      row    = sheet.getRow(i);
      prefix = "";
      if (prefixCols.length > 0) {
	for (n = 0; n < prefixCols.length; n++) {
	  if (n > 0)
	    prefix += " ";  // TODO option?
	  if (row.hasCell(prefixCols[n]))
	    prefix += row.getCell(prefixCols[n]).getContent();
	  else
	    prefix += SpreadSheet.MISSING_VALUE;
	}
	prefix += " ";
      }
      for (n = 0; n < plotCols.length; n++) {
	if (xIndex == -1) {
	  cont = new SequencePlotterContainer(prefix + plotNames[n], new Double(i), getCellValue(row, plotCols[n]));
	}
	else {
	  if (xIndex == plotCols[n])
	    continue;
	  if (xIndexString == null) {
	    cont = new SequencePlotterContainer(prefix + plotNames[n], getCellValue(row, xIndex), getCellValue(row, plotCols[n]));
	  }
	  else {
	    if (!xIndexString.containsKey(row.getCell(xIndex).getContent()))
	      xIndexString.put(row.getCell(xIndex).getContent(), (xIndexString.size() + 1));
	    cont = new SequencePlotterContainer(prefix + plotNames[n], (double) xIndexString.get(row.getCell(xIndex).getContent()), getCellValue(row, plotCols[n]));
	  }
	}
	m_Containers.add(cont);
      }
    }

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.flow.container.SequencePlotterContainer.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{SequencePlotterContainer.class};
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  @Override
  public boolean hasPendingOutput() {
    return (m_Containers.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  @Override
  public Token output() {
    Token	result;

    result        = new Token(m_Containers.get(0));
    m_InputToken  = null;
    m_Containers.remove(0);

    return result;
  }
}
