/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetColumnIndex.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.data.spreadsheet;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import adams.core.Index;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;

/**
 * Extended {@link Index} class that can use a column name to determine an
 * index of a column as well.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6866 $
 */
public class SpreadSheetColumnIndex
  extends Index {

  /** for serialization. */
  private static final long serialVersionUID = -4358263779315198808L;

  /** the underlying spreadsheet. */
  protected SpreadSheet m_Sheet;
  
  /** the column names to replace. */
  protected List<String> m_Names;
  
  /** the indices of the column names. */
  protected HashMap<String,Integer> m_Indices;
  
  /**
   * Initializes with no index.
   */
  public SpreadSheetColumnIndex() {
    super();
  }

  /**
   * Initializes with the given index, but no maximum.
   *
   * @param index	the index to use
   */
  public SpreadSheetColumnIndex(String index) {
    super(index);
  }

  /**
   * Initializes with the given index and maximum.
   *
   * @param index	the index to use
   * @param max		the maximum of the 1-based index (e.g., use "10" to
   * 			allow "1-10" or -1 for uninitialized)
   */
  public SpreadSheetColumnIndex(String index, int max) {
    super(index, max);
  }
  
  /**
   * For initializing the object.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_Sheet   = null;
    m_Names   = null;
    m_Indices = null;
  }
  
  /**
   * Sets the spreadsheet to use for interpreting the column name.
   * 
   * @param value	the spreadsheet to use, can be null
   */
  public void setSpreadSheet(SpreadSheet value) {
    m_Sheet = value;
    m_Names = null;
    if (m_Sheet == null)
      setMax(-1);
    else
      setMax(value.getColumnCount());
  }
  
  /**
   * Returns the underlying spreadsheet.
   * 
   * @return		the underlying spreadsheet, null if none set
   */
  public SpreadSheet getSpreadSheet() {
    return m_Sheet;
  }

  /**
   * Returns the column names.
   * 
   * @return		the column names
   */
  protected synchronized List<String> getNames() {
    Row		header;
    int		i;
    String	col;
    
    if (m_Names == null) {
      m_Names   = new ArrayList<String>();
      m_Indices = new HashMap<String,Integer>();
      header    = m_Sheet.getHeaderRow();
      for (i = 0; i < m_Sheet.getColumnCount(); i++) {
	col = header.getCell(i).getContent();
        m_Names.add(col);
        m_Indices.put(col, i);
      }
      Collections.sort(m_Names);
      Collections.reverse(m_Names);
    }
    
    return m_Names;
  }
  
  /**
   * Returns the indices in use.
   * 
   * @return		the indices
   */
  public synchronized HashMap<String,Integer> getIndices() {
    if (m_Names == null)
      getNames();
    return m_Indices;
  }
  
  /**
   * Replaces any column name in the string with the actual 1-based index.
   * 
   * @param s		the string to process
   * @return		the (potentially) updated string
   */
  protected String replaceColumnName(String s) {
    String		result;
    int			i;
    List<String>	names;
    
    result = s.trim();
    
    names = getNames();
    for (i = 0; i < names.size(); i++) {
      if (result.equals(names.get(i))) {
	result = Integer.toString(getIndices().get(names.get(i)) + 1);
	break;
      }
    }
    
    return result;
  }
  
  /**
   * Checks whether the strings represents a column name.
   * 
   * @param s		the string to process
   * @return		true if string is a column name
   */
  protected boolean isColumnName(String s) {
    boolean		result;
    int			i;
    List<String>	names;
    
    result = false;
    
    names = getNames();
    for (i = 0; i < names.size(); i++) {
      if (s.equals(names.get(i))) {
	result = true;
	break;
      }
    }
    
    return result;
  }
  
  /**
   * Cleanses the given string. Only allows "first", "last" and numbers.
   *
   * @param s		the string to clean
   * @return		the cleansed string, "" if invalid one provided; if 
   * 			no spreadsheet provided, all input is considered valid
   */
  @Override
  protected String clean(String s) {
    if (m_Sheet == null) {
      if (isPlaceholder(s))
	return super.clean(s);
      else
	return s;
    }
    else {
      if (isColumnName(s))
	return s;
      else
	return super.clean(s);
    }
  }
  
  /**
   * Parses the string and checks it against the maximum.
   *
   * @param s		the string to parse
   * @param max		the maximum to allow
   * @return		the parsed value, -1 if invalid
   */
  @Override
  protected int parse(String s, int max) {
    if (m_Sheet == null)
      return super.parse(s, max);
    else
      return super.parse(replaceColumnName(s), max);
  }

  /**
   * Returns the example.
   *
   * @return		the example
   */
  @Override
  public String getExample() {
    return
        "An index is a number starting with 1; apart from column names "
      + "(case-sensitive), the following placeholders can be used as well: "
      + FIRST + ", " + SECOND + ", " + THIRD + ", " + LAST_2 + ", " + LAST_1 + ", " + LAST;
  }
}
