/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetColumnRange.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.data.spreadsheet;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import adams.core.Range;

/**
 * Extended {@link Range} class that also allows column names for specifying
 * column positions (names are case-insensitive, just like placeholders for 
 * 'first', 'second', etc). If column names contain "-" or "," then they
 * need to be surrounded by double-quotes.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6867 $
 */
public class SpreadSheetColumnRange
  extends Range {

  /** for serialization. */
  private static final long serialVersionUID = 5215987200366396733L;

  /** the underlying spreadsheet. */
  protected SpreadSheet m_Sheet;
  
  /** the column names to replace. */
  protected List<String> m_Names;
  
  /** the indices of the column names. */
  protected HashMap<String,Integer> m_Indices;

  /**
   * Initializes with no range.
   */
  public SpreadSheetColumnRange() {
    super();
  }

  /**
   * Initializes with the given range, but no maximum.
   *
   * @param range	the range to use
   */
  public SpreadSheetColumnRange(String range) {
    super(range);
  }

  /**
   * Initializes with the given range and maximum.
   *
   * @param range	the range to use
   * @param max		the maximum of the 1-based index (e.g., use "10" to
   * 			allow "1-10" or -1 for uninitialized)
   */
  public SpreadSheetColumnRange(String range, int max) {
    super(range, max);
  }
  
  /**
   * For initializing the object.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_Sheet   = null;
    m_Names   = null;
    m_Indices = null;
  }
  
  /**
   * Sets the spreadsheet to use for interpreting the column name.
   * 
   * @param value	the spreadsheet to use, can be null
   */
  public void setSpreadSheet(SpreadSheet value) {
    m_Sheet = value;
    m_Names = null;
    if (m_Sheet == null)
      setMax(-1);
    else
      setMax(value.getColumnCount());
  }
  
  /**
   * Returns the underlying spreadsheet.
   * 
   * @return		the underlying spreadsheet, null if none set
   */
  public SpreadSheet getSpreadSheet() {
    return m_Sheet;
  }

  /**
   * Returns the column names.
   * 
   * @return		the column names
   */
  protected synchronized List<String> getNames() {
    Row		header;
    int		i;
    String	col;
    
    if (m_Names == null) {
      m_Names = new ArrayList<String>();
      if (m_Sheet != null) {
	m_Indices = new HashMap<String,Integer>();
	header    = m_Sheet.getHeaderRow();
	for (i = 0; i < m_Sheet.getColumnCount(); i++) {
	  col = header.getCell(i).getContent();
	  if (col.indexOf(RANGE) > -1)
	    col = "\"" + col + "\"";
	  else if (col.indexOf(SEPARATOR) > -1)
	    col = "\"" + col + "\"";
	  m_Names.add(col);
	  m_Indices.put(col, i);
	}
	Collections.sort(m_Names);
	Collections.reverse(m_Names);
      }
    }
    
    return m_Names;
  }
  
  /**
   * Returns the indices in use.
   * 
   * @return		the indices
   */
  public synchronized HashMap<String,Integer> getIndices() {
    if (m_Names == null)
      getNames();
    return m_Indices;
  }
  
  /**
   * Checks whether the strings represents a column name.
   * 
   * @param s		the string to process
   * @return		true if string is a column name
   */
  protected boolean isColumnName(String s) {
    boolean		result;
    int			i;
    List<String>	names;
    
    result = false;
    
    names = getNames();
    for (i = 0; i < names.size(); i++) {
      if (s.equals(names.get(i))) {
	result = true;
	break;
      }
    }
    
    return result;
  }

  /**
   * Returns the placeholders to allow in the ranges. This includes the
   * column names.
   * 
   * @return		the placeholders
   * @see		#getNames()
   */
  @Override
  protected List<String> getPlaceholders() {
    List<String>	result;
    
    result = super.getPlaceholders();
    result.addAll(getNames());
    
    return result;
  }
  
  /**
   * Returns whether invalid characters should get removed.
   * 
   * @return		true if to replace invalid chars
   */
  @Override
  protected boolean canReplaceInvalidChars() {
    return (m_Sheet != null);
  }
  
  /**
   * Attempts to split a range into the parts resembling it.
   * 
   * @param s		the string to split
   * @return		the parts (single array element if no range)
   */
  @Override
  protected String[] splitRange(String s) {
    return SpreadSheetUtils.split(s, RANGE.charAt(0));
  }
  
  /**
   * Attempts to split a list into the parts resembling it.
   * 
   * @param s		the string to split
   * @return		the parts (single array element if no list)
   */
  @Override
  protected String[] splitList(String s) {
    return SpreadSheetUtils.split(s, SEPARATOR.charAt(0));
  }

  /**
   * Parses the 1-based index, 'first' and 'last' are accepted as well.
   *
   * @param s		the string to parse
   * @param max		the maximum value to use
   * @return		the 0-based index
   */
  @Override
  protected int parse(String s, int max) {
    if (isColumnName(s))
      return getIndices().get(s);
    else
      return super.parse(s, max);
  }
}
