/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * XYPlotGenerator.java
 * Copyright (C) 2012-2013 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer.plotgenerator;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import adams.data.spreadsheet.Cell.ContentType;
import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;
import adams.data.spreadsheet.SpreadSheetColumnIndex;
import adams.data.spreadsheet.SpreadSheetColumnRange;
import adams.flow.container.SequencePlotterContainer;

/**
 <!-- globalinfo-start -->
 * Uses one column for the X value and one or more other columns as Y to plot against.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-default-cell-value &lt;double&gt; (property: defaultCellValue)
 * &nbsp;&nbsp;&nbsp;The default value for missing or non-numeric cells.
 * &nbsp;&nbsp;&nbsp;default: -1.0
 * </pre>
 * 
 * <pre>-columns &lt;java.lang.String&gt; (property: plotColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to include in the plot; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-x-column &lt;java.lang.String&gt; (property: XColumn)
 * &nbsp;&nbsp;&nbsp;The (optional) index of the column which values to use as X values in the 
 * &nbsp;&nbsp;&nbsp;plot; An index is a number starting with 1; apart from column names (case-sensitive
 * &nbsp;&nbsp;&nbsp;), the following placeholders can be used as well: first, second, third, 
 * &nbsp;&nbsp;&nbsp;last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-prefix-columns &lt;adams.data.spreadsheet.SpreadSheetColumnRange&gt; (property: prefixColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to prefix the plot name with; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-separator &lt;java.lang.String&gt; (property: separator)
 * &nbsp;&nbsp;&nbsp;The separator to use between columns.
 * &nbsp;&nbsp;&nbsp;default:  
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6969 $
 */
public class XYPlotGenerator
  extends AbstractPlotGenerator {

  /** for serialization. */
  private static final long serialVersionUID = 2881757430571628883L;

  /** the columns to plot. */
  protected SpreadSheetColumnRange m_PlotColumns;

  /** the (optional) column to use as X value. */
  protected SpreadSheetColumnIndex m_XColumn;
  
  /** the columns to prefix the plot name with. */
  protected SpreadSheetColumnRange m_PrefixColumns;
  
  /** the separator string. */
  protected String m_Separator;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Uses one column for the X value and one or more other columns as Y to plot against.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "plotColumns",
	    "");

    m_OptionManager.add(
	    "x-column", "XColumn",
	    "");

    m_OptionManager.add(
	    "prefix-columns", "prefixColumns",
	    new SpreadSheetColumnRange());

    m_OptionManager.add(
	    "separator", "separator",
	    " ");
  }

  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_PlotColumns = new SpreadSheetColumnRange();
    m_XColumn     = new SpreadSheetColumnIndex();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "x: ";
    variable = getOptionManager().getVariableForProperty("XColumn");
    if (variable != null)
      result += variable;
    else
      result += m_XColumn.getIndex();
    
    result   += ", y cols: ";
    variable = getOptionManager().getVariableForProperty("plotColumns");
    if (variable != null)
      result += variable;
    else
      result += m_PlotColumns.getRange();
    
    result   += ", prefix cols: ";
    variable = getOptionManager().getVariableForProperty("prefixColumns");
    if (variable != null)
      result += variable;
    else
      result += m_PrefixColumns.getRange();
    
    result   += ", separator: ";
    variable = getOptionManager().getVariableForProperty("separator");
    if (variable != null)
      result += variable;
    else
      result += m_Separator;
    
    return result;
  }

  /**
   * Sets the column range to use in the plot.
   *
   * @param value	the column range
   */
  public void setPlotColumns(String value) {
    m_PlotColumns.setRange(value);
    reset();
  }

  /**
   * Returns the current column range to use in the plot.
   *
   * @return		the column range
   */
  public String getPlotColumns() {
    return m_PlotColumns.getRange();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String plotColumnsTipText() {
    return "The range of columns to include in the plot; " + m_PlotColumns.getExample();
  }

  /**
   * Sets the index of the column which values to use as X values.
   *
   * @param value	the column index
   */
  public void setXColumn(String value) {
    m_XColumn.setIndex(value);
    reset();
  }

  /**
   * Returns the index of the column which values to use as X values.
   *
   * @return		the column index
   */
  public String getXColumn() {
    return m_XColumn.getIndex();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String XColumnTipText() {
    return "The (optional) index of the column which values to use as X values in the plot; " + m_XColumn.getExample();
  }

  /**
   * Sets the column range to generate the prefix of the plot with.
   *
   * @param value	the column range
   */
  public void setPrefixColumns(SpreadSheetColumnRange value) {
    m_PrefixColumns = value;
    reset();
  }

  /**
   * Returns the current column range to generate the prefix of the plot with.
   *
   * @return		the column range
   */
  public SpreadSheetColumnRange getPrefixColumns() {
    return m_PrefixColumns;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String prefixColumnsTipText() {
    return "The range of columns to prefix the plot name with; " + m_PlotColumns.getExample();
  }

  /**
   *Missing tooltip: adams.flow.transformer.plotgenerator.XYPlotGenerator.separatorTipText
 Sets the separator to use between columns.
   *
   * @param value	the separator
   */
  public void setSeparator(String value) {
    m_Separator = value;
    reset();
  }

  /**
   * Returns the separator in use between columns.
   *
   * @return		the separator
   */
  public String getSeparator() {
    return m_Separator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String separatorTipText() {
    return "The separator to use between columns.";
  }

  /**
   * Checks the spreadsheet.
   * 
   * @param sheet	the sheet to check
   */
  @Override
  protected void check(SpreadSheet sheet) {
    super.check(sheet);

    m_PlotColumns.setSpreadSheet(sheet);
    if (m_PlotColumns.getIntIndices().length == 0)
      throw new IllegalStateException("No plot columns defined/available?");
    
    m_XColumn.setSpreadSheet(sheet);
    if (m_XColumn.getIntIndex() == -1)
      throw new IllegalStateException("No X column defined/available?");

    m_PrefixColumns.setSpreadSheet(sheet);
  }
  
  /**
   * Performs the actual generation of containers.
   * 
   * @param sheet	the basis for the containers
   * @return		the generated containers
   */
  @Override
  protected List<SequencePlotterContainer> doGenerate(SpreadSheet sheet) {
    ArrayList<SequencePlotterContainer>	result;
    SequencePlotterContainer		cont;
    int					i;
    int					n;
    Row					row;
    int[]				plotCols;
    String[]				plotNames;
    int					xIndex;
    int[]				prefixCols;
    String				prefix;
    Hashtable<String,Integer>		xIndexString;

    result = new ArrayList<SequencePlotterContainer>();

    // gather data
    xIndex       = m_XColumn.getIntIndex();
    xIndexString = null;
    if (xIndex != -1) {
      if (! (   sheet.isNumeric(xIndex) 
	     || sheet.isContentType(xIndex, ContentType.TIME) 
	     || sheet.isContentType(xIndex, ContentType.DATE) 
	     || sheet.isContentType(xIndex, ContentType.DATETIME)))
	xIndexString = new Hashtable<String,Integer>();
    }

    // create plot containers
    plotCols   = m_PlotColumns.getIntIndices();
    plotNames  = new String[plotCols.length];
    prefixCols = m_PrefixColumns.getIntIndices();
    for (i = 0; i < plotNames.length; i++)
      plotNames[i] = sheet.getHeaderRow().getCell(plotCols[i]).getContent();
    for (i = 0; i < sheet.getRowCount(); i++) {
      row    = sheet.getRow(i);
      prefix = "";
      if (prefixCols.length > 0) {
	for (n = 0; n < prefixCols.length; n++) {
	  if (n > 0)
	    prefix += m_Separator;
	  if (row.hasCell(prefixCols[n]))
	    prefix += row.getCell(prefixCols[n]).getContent();
	  else
	    prefix += SpreadSheet.MISSING_VALUE;
	}
	prefix += " ";
      }
      for (n = 0; n < plotCols.length; n++) {
	if (xIndex == -1) {
	  cont = new SequencePlotterContainer(prefix + plotNames[n], new Double(i), getCellValue(row, plotCols[n]));
	}
	else {
	  if (xIndex == plotCols[n])
	    continue;
	  if (xIndexString == null) {
	    cont = new SequencePlotterContainer(prefix + plotNames[n], getCellValue(row, xIndex), getCellValue(row, plotCols[n]));
	  }
	  else {
	    if (!xIndexString.containsKey(row.getCell(xIndex).getContent()))
	      xIndexString.put(row.getCell(xIndex).getContent(), (xIndexString.size() + 1));
	    cont = new SequencePlotterContainer(prefix + plotNames[n], (double) xIndexString.get(row.getCell(xIndex).getContent()), getCellValue(row, plotCols[n]));
	  }
	}
	result.add(cont);
      }
    }

    return result;
  }
}
