/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * TabbedPane.java
 * Copyright (C) 2009-2013 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.tools.spreadsheetviewer;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import javax.swing.JScrollPane;

import adams.data.spreadsheet.SpreadSheet;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseTabbedPane;
import adams.gui.core.SpreadSheetTable;

/**
 * A specialized tabbed pane with a few methods for easier access.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6951 $
 */
public class TabbedPane
  extends BaseTabbedPane {

  /** for serialization. */
  private static final long serialVersionUID = -2048229771213837710L;

  /** prefix for new titles. */
  public final static String PREFIX_TITLE = "new";
  
  /**
   * Returns the table at the specified position.
   *
   * @param index	the tab index of the table
   * @return		the table, null if not available or invalid index
   */
  public SpreadSheetTable getTableAt(int index) {
    SpreadSheetTable	result;
    JScrollPane		pane;

    if ((index < 0) || (index >= getTabCount()))
      return null;
    
    if (getComponentAt(index) instanceof JScrollPane) {
      pane   = (JScrollPane) getComponentAt(index);
      result = (SpreadSheetTable) pane.getViewport().getView();
    }
    else {
      result = (SpreadSheetTable) getComponentAt(index);
    }

    return result;
  }
  
  /**
   * Returns the table at the specified position.
   *
   * @param index	the tab index of the table
   * @return		the table
   */
  public SpreadSheet getSheetAt(int index) {
    SpreadSheet		result;
    SpreadSheetTable	table;
    
    result = null;
    table  = getTableAt(index);
    if (table == null)
      return result;

    result = table.toSpreadSheet();

    return result;
  }

  /**
   * Sets the number of decimals to use.
   *
   * @param index	the tab index
   * @param numDec	the number of decimals to use
   */
  public void setNumDecimalsAt(int index, int numDec) {
    getTableAt(index).setNumDecimals(numDec);
  }

  /**
   * returns the number of decimals in use.
   *
   * @param index	the tab index
   * @return		the number of decimals in use
   */
  public int getNumDecimalsAt(int index) {
    return getTableAt(index).getNumDecimals();
  }

  /**
   * Sets the number of decimals to use for all tables.
   *
   * @param numDec	the number of decimals to use
   */
  public void setNumDecimals(int numDec) {
    int	i;

    for (i = 0; i < getTabCount(); i++)
      setNumDecimalsAt(i, numDec);
  }

  /**
   * Sets the background color to use negative values.
   *
   * @param index	the tab index
   * @param color	the color to use
   */
  public void setNegativeBackgroundAt(int index, Color color) {
    getTableAt(index).setNegativeBackground(color);
  }

  /**
   * Returns the background color in use for negative values.
   *
   * @param index	the tab index
   * @return		the color to use, null if none set
   */
  public Color getNegativeBackgroundAt(int index) {
    return getTableAt(index).getNegativeBackground();
  }

  /**
   * Sets the background color to use for negative values.
   *
   * @param color	the color to use, null to unset
   */
  public void setNegativeBackground(Color color) {
    int	i;

    for (i = 0; i < getTabCount(); i++)
      setNegativeBackgroundAt(i, color);
  }

  /**
   * Sets the background color to use positive values.
   *
   * @param index	the tab index
   * @param color	the color to use
   */
  public void setPositiveBackgroundAt(int index, Color color) {
    getTableAt(index).setPositiveBackground(color);
  }

  /**
   * Returns the background color in use for positive values.
   *
   * @param index	the tab index
   * @return		the color to use, null if none set
   */
  public Color getPositiveBackgroundAt(int index) {
    return getTableAt(index).getPositiveBackground();
  }

  /**
   * Sets the background color to use for positive values.
   *
   * @param color	the color to use, null to unset
   */
  public void setPositiveBackground(Color color) {
    int	i;

    for (i = 0; i < getTabCount(); i++)
      setPositiveBackgroundAt(i, color);
  }

  /**
   * Sets whether to show the formulas.
   *
   * @param index	the tab index
   * @param value	whether to show the formulas
   */
  public void setShowFormulasAt(int index, boolean value) {
    getTableAt(index).setShowFormulas(value);
  }

  /**
   * Returns whether to show the formulas.
   *
   * @param index	the tab index
   * @return		whether to show the formulas
   */
  public boolean getShowFormulas(int index) {
    return getTableAt(index).getShowFormulas();
  }

  /**
   * Sets whether to show the formulas.
   *
   * @param value	whether to show the formulas
   */
  public void setShowFormulas(boolean value) {
    int	i;

    for (i = 0; i < getTabCount(); i++)
      setShowFormulasAt(i, value);
  }

  /**
   * Returns the currently selected table.
   * 
   * @return		the table, null if none available
   */
  public SpreadSheetTable getCurrentTable() {
    return getTableAt(getSelectedIndex());
  }

  /**
   * Returns the currently selected sheet.
   * 
   * @return		the sheet, null if none available
   */
  public SpreadSheet getCurrentSheet() {
    return getSheetAt(getSelectedIndex());
  }
  
  /**
   * Adds the table.
   * 
   * @param title	the title for the tab
   * @param table	the table to add
   */
  public void addTab(String title, SpreadSheetTable table) {
    addTab(title, new BaseScrollPane(table));
    setSelectedIndex(getTabCount() - 1);
  }
  
  /**
   * Returns all the tab titles.
   * 
   * @return		the titles
   */
  public List<String> getTabTitles() {
    ArrayList<String>	result;
    int			i;
    
    result = new ArrayList<String>();
    
    for (i = 0; i < getTabCount(); i++)
      result.add(getTitleAt(i));
    
    return result;
  }
  
  /**
   * Returns a new title (does not reserve it).
   */
  public String newTitle() {
    String		result;
    int			i;
    HashSet<String>	titles;
    
    titles = new HashSet<String>(getTabTitles());
    i      = 0;
    do {
      i++;
      result = PREFIX_TITLE + i;
    }
    while (titles.contains(result));
    
    return result;
  }
}