/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetConvertCells.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer;

import java.util.Date;
import java.util.Iterator;

import adams.core.DateTime;
import adams.core.QuickInfoHelper;
import adams.core.Time;
import adams.data.conversion.Conversion;
import adams.data.conversion.ObjectToObject;
import adams.data.spreadsheet.Cell;
import adams.data.spreadsheet.SpreadSheet;
import adams.data.spreadsheet.cellfinder.AbstractCellFinder;
import adams.data.spreadsheet.cellfinder.CellLocation;
import adams.data.spreadsheet.cellfinder.CellRange;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Finds cells in a spreadsheet and converts them with a conversion scheme.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetConvertCells
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-no-copy (property: noCopy)
 * &nbsp;&nbsp;&nbsp;If enabled, no copy of the spreadsheet is created before processing it.
 * </pre>
 * 
 * <pre>-finder &lt;adams.data.spreadsheet.cellfinder.AbstractCellFinder&gt; (property: finder)
 * &nbsp;&nbsp;&nbsp;The cell finder to use.
 * &nbsp;&nbsp;&nbsp;default: adams.data.spreadsheet.cellfinder.CellRange
 * </pre>
 * 
 * <pre>-conversion &lt;adams.data.conversion.Conversion&gt; (property: conversion)
 * &nbsp;&nbsp;&nbsp;The conversion to apply to the located cells.
 * &nbsp;&nbsp;&nbsp;default: adams.data.conversion.ObjectToObject
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 7726 $
 */
public class SpreadSheetConvertCells
  extends AbstractInPlaceSpreadSheetTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -4633161214275622241L;

  /** for locating the cells. */
  protected AbstractCellFinder m_Finder;
  
  /** the conversion to apply to the cells. */
  protected Conversion m_Conversion;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Finds cells in a spreadsheet and converts them with a conversion scheme.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "finder", "finder",
	    new CellRange());

    m_OptionManager.add(
	    "conversion", "conversion",
	    new ObjectToObject());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;

    result  = QuickInfoHelper.toString(this, "finder", m_Finder, "finder: ");
    result += QuickInfoHelper.toString(this, "conversion", m_Conversion, ", conversion: ");
    result += QuickInfoHelper.toString(this, "noCopy", m_NoCopy, "no copy", ", ");

    return result;
  }

  /**
   * Sets the cell finder to use.
   *
   * @param value	the finder
   */
  public void setFinder(AbstractCellFinder value) {
    m_Finder = value;
    reset();
  }

  /**
   * Returns the cell finder to use.
   *
   * @return		the finder
   */
  public AbstractCellFinder getFinder() {
    return m_Finder;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String finderTipText() {
    return "The cell finder to use.";
  }

  /**
   * Sets the conversion scheme to apply to the located cells.
   *
   * @param value	the conversion
   */
  public void setConversion(Conversion value) {
    m_Conversion = value;
    reset();
  }

  /**
   * Returns the conversion scheme to apply to the located cells.
   *
   * @return		the conversion
   */
  public Conversion getConversion() {
    return m_Conversion;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String conversionTipText() {
    return "The conversion to apply to the located cells.";
  }
  
  /**
   * Applies the conversion to the cell.
   * 
   * @param location	the cell location to convert
   * @param sheet	the sheet to process
   * @return		null if successful, otherwise error message
   */
  protected String convertCell(CellLocation location, SpreadSheet sheet) {
    String	result;
    Cell	cell;
    Class	classIn;
    Class	classOut;
    Object	input;
    Object	output;
    
    if (!sheet.hasCell(location.getRow(), location.getColumn()))
      return null;
    
    result   = null;
    
    cell     = sheet.getCell(location.getRow(), location.getColumn());
    input    = null;
    output   = null;
    classIn  = m_Conversion.accepts();
    classOut = m_Conversion.generates();
    
    if (classIn == Double.class)
      input = cell.toDouble();
    else if (classIn == Integer.class)
      input = cell.toLong().intValue();
    else if (classIn == Long.class)
      input = cell.toLong();
    else if (classIn == Date.class)
      input = cell.toDate();
    else if (classIn == DateTime.class)
      input = cell.toDateTime();
    else if (classIn == Time.class)
      input = cell.toTime();
    else if (classIn == String.class)
      input = cell.getContent();
    else
      result = "Don't know how to get cell value for conversion input type: " + classIn.getName();
    
    if (result == null) {
      m_Conversion.setInput(input);
      result = m_Conversion.convert();
    }
      
    if (result == null) {
      output = m_Conversion.getOutput();
      m_Conversion.cleanUp();
      
      if (classOut == Double.class)
	cell.setContent((Double) output);
      else if (classOut == Integer.class)
	cell.setContent((Integer) output);
      else if (classOut == Long.class)
	cell.setContent((Long) output);
      else if (classOut == Date.class)
	cell.setContent((Date) output);
      else if (classOut == DateTime.class)
	cell.setContent((DateTime) output);
      else if (classOut == Time.class)
	cell.setContent((Time) output);
      else if (classOut == String.class)
	cell.setContent((String) output);
      else
	result = "Don't know how to set cell value for conversion output type: " + classOut.getName();
    }
    
    return result;
  }
  
  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String			result;
    SpreadSheet			sheetOld;
    SpreadSheet			sheetNew;
    Iterator<CellLocation>	cells;
    
    result = null;
    
    sheetOld = (SpreadSheet) m_InputToken.getPayload();
    if (m_NoCopy)
      sheetNew = sheetOld;
    else
      sheetNew = sheetOld.getClone();
    
    cells = m_Finder.findCells(sheetNew);
    while (cells.hasNext() && !m_Stopped) {
      result = convertCell(cells.next(), sheetNew);
      if (result != null)
	break;
    }
    
    if (result == null)
      m_OutputToken = new Token(sheetNew);
    
    return result;
  }
  
  /**
   * Stops the execution. No message set.
   */
  @Override
  public void stopExecution() {
    super.stopExecution();
    m_Conversion.stopExecution();
  }
}
