/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetTransformCells.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer;

import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import adams.core.DateTime;
import adams.core.QuickInfoHelper;
import adams.core.Time;
import adams.core.Utils;
import adams.data.spreadsheet.Cell;
import adams.data.spreadsheet.SpreadSheet;
import adams.data.spreadsheet.cellfinder.AbstractCellFinder;
import adams.data.spreadsheet.cellfinder.CellLocation;
import adams.data.spreadsheet.cellfinder.CellRange;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorUtils;
import adams.flow.core.Compatibility;
import adams.flow.core.GlobalActorHelper;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.GlobalActorUser;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Finds cells in a spreadsheet and transforms them with a global transformer.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.spreadsheet.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SpreadSheetTransformCells
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-no-copy (property: noCopy)
 * &nbsp;&nbsp;&nbsp;If enabled, no copy of the spreadsheet is created before processing it.
 * </pre>
 * 
 * <pre>-finder &lt;adams.data.spreadsheet.cellfinder.AbstractCellFinder&gt; (property: finder)
 * &nbsp;&nbsp;&nbsp;The cell finder to use.
 * &nbsp;&nbsp;&nbsp;default: adams.data.spreadsheet.cellfinder.CellRange
 * </pre>
 * 
 * <pre>-transformer &lt;adams.flow.core.GlobalActorReference&gt; (property: transformer)
 * &nbsp;&nbsp;&nbsp;The global transformer to apply to the located cells.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 8073 $
 */
public class SpreadSheetTransformCells
  extends AbstractInPlaceSpreadSheetTransformer 
  implements GlobalActorUser {

  /** for serialization. */
  private static final long serialVersionUID = -2533024333073383813L;

  /** the key for backing up the global actor. */
  public final static String BACKUP_GLOBALACTOR = "global actor";

  /** for locating the cells. */
  protected AbstractCellFinder m_Finder;
  
  /** the global transformer to apply to the cells. */
  protected GlobalActorReference m_Transformer;

  /** the helper class. */
  protected GlobalActorHelper m_Helper;

  /** the global actor. */
  protected AbstractActor m_GlobalActor;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return "Finds cells in a spreadsheet and transforms them with a global transformer.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "finder", "finder",
	    new CellRange());

    m_OptionManager.add(
	    "transformer", "transformer",
	    new GlobalActorReference("unknown"));
  }
  
  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_Helper = new GlobalActorHelper();
  }

  /**
   * Resets the scheme.
   */
  @Override
  protected void reset() {
    super.reset();
    
    m_GlobalActor = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	result;

    result  = QuickInfoHelper.toString(this, "finder", m_Finder, "finder: ");
    result += QuickInfoHelper.toString(this, "transformer", m_Transformer, ", transformer: ");
    result += QuickInfoHelper.toString(this, "noCopy", m_NoCopy, "no copy", ", ");

    return result;
  }

  /**
   * Sets the cell finder to use.
   *
   * @param value	the finder
   */
  public void setFinder(AbstractCellFinder value) {
    m_Finder = value;
    reset();
  }

  /**
   * Returns the cell finder to use.
   *
   * @return		the finder
   */
  public AbstractCellFinder getFinder() {
    return m_Finder;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String finderTipText() {
    return "The cell finder to use.";
  }

  /**
   * Sets the reference to the global transformer.
   *
   * @param value	the reference
   */
  public void setTransformer(GlobalActorReference value) {
    m_Transformer = value;
    reset();
  }

  /**
   * Returns the reference to the global transformer.
   *
   * @return		the conversion
   */
  public GlobalActorReference getTransformer() {
    return m_Transformer;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String transformerTipText() {
    return "The global transformer to apply to the located cells.";
  }

  /**
   * Tries to find the global actor referenced by its global name.
   *
   * @return		the global actor or null if not found
   */
  protected AbstractActor findGlobalActor() {
    return m_Helper.findGlobalActorRecursive(this, getTransformer());
  }

  /**
   * Checks whether a reference to the global actor is currently available.
   *
   * @return		true if a reference is available
   * @see		#getGlobalActor()
   */
  public boolean hasGlobalActor() {
    return (m_GlobalActor != null);
  }

  /**
   * Returns the currently set global actor.
   *
   * @return		the actor, can be null
   */
  @Override
  public AbstractActor getGlobalActor() {
    return m_GlobalActor;
  }

  /**
   * Removes entries from the backup.
   */
  @Override
  protected void pruneBackup() {
    super.pruneBackup();
    pruneBackup(BACKUP_GLOBALACTOR);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  @Override
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;
    
    result = super.backupState();
    
    if (m_GlobalActor != null)
      result.put(BACKUP_GLOBALACTOR, m_GlobalActor);
    
    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  @Override
  protected void restoreState(Hashtable<String,Object> state) {
    super.restoreState(state);
    
    if (state.containsKey(BACKUP_GLOBALACTOR)) {
      m_GlobalActor = (AbstractActor) state.get(BACKUP_GLOBALACTOR);
      state.remove(BACKUP_GLOBALACTOR);
    }
  }

  /**
   * Configures the global actor.
   * 
   * @return		null if successful, otherwise error message
   */
  protected String setUpGlobalActor() {
    String		result;
    HashSet<String>	variables;

    result = null;

    m_GlobalActor = findGlobalActor();
    if (m_GlobalActor == null) {
      result = "Couldn't find global transformer '" + getTransformer() + "'!";
    }
    else {
      if (ActorUtils.isTransformer(m_GlobalActor)) {
	variables = findVariables(m_GlobalActor);
	m_DetectedVariables.addAll(variables);
	if (m_DetectedVariables.size() > 0)
	  getVariables().addVariableChangeListener(this);
      }
      else {
	result = "Global actor '" + getTransformer() + "' is not a transformer!";
      }
    }

    return result;
  }
  
  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  public String setUp() {
    String	result;
    String	variable;

    result = super.setUp();

    if (result == null) {
      // do we have to wait till execution time because of attached variable?
      variable = getOptionManager().getVariableForProperty("transformer");
      if (variable == null)
	result = setUpGlobalActor();
    }

    return result;
  }
  
  /**
   * Applies the conversion to the cell.
   * 
   * @param location	the cell location to convert
   * @param sheet	the sheet to process
   * @return		null if successful, otherwise error message
   */
  protected String transformCell(CellLocation location, SpreadSheet sheet) {
    String		result;
    Cell		cell;
    Class[]		classIn;
    Class[]		classOut;
    Object		input;
    Object		output;
    Compatibility	comp;
    
    if (!sheet.hasCell(location.getRow(), location.getColumn()))
      return null;
    
    // skip missing cells
    cell = sheet.getCell(location.getRow(), location.getColumn());
    if (cell.isMissing())
      return null;
    
    result   = null;
    input    = null;
    output   = null;
    classIn  = ((InputConsumer) m_GlobalActor).accepts();
    classOut = ((OutputProducer) m_GlobalActor).generates();
    comp     = new Compatibility();
    
    if (comp.isCompatible(new Class[]{Double.class}, classIn))
      input = cell.toDouble();
    else if (comp.isCompatible(new Class[]{Integer.class}, classIn))
      input = cell.toLong().intValue();
    else if (comp.isCompatible(new Class[]{Long.class}, classIn))
      input = cell.toLong();
    else if (comp.isCompatible(new Class[]{Date.class}, classIn))
      input = cell.toDate();
    else if (comp.isCompatible(new Class[]{DateTime.class}, classIn))
      input = cell.toDateTime();
    else if (comp.isCompatible(new Class[]{Time.class}, classIn))
      input = cell.toTime();
    else if (comp.isCompatible(new Class[]{String.class}, classIn))
      input = cell.getContent();
    else
      result = "Don't know how to get cell value for conversion input type:\n" 
	  + Utils.arrayToString(classIn)
	  + "/" 
	  + ((input != null) ? input.getClass().getName() : "null")
	  + "/" + cell.getContent();

    if (result == null) {
      ((InputConsumer) m_GlobalActor).input(new Token(input));
      result = m_GlobalActor.execute();
    }
      
    if (result == null) {
      output = ((OutputProducer) m_GlobalActor).output();
      if (output != null)
	output = ((Token) output).getPayload();
      
      if (output instanceof Double)
	cell.setContent((Double) output);
      else if (output instanceof Integer)
	cell.setContent((Integer) output);
      else if (output instanceof Long)
	cell.setContent((Long) output);
      else if (output instanceof DateTime)
	cell.setContent((DateTime) output);
      else if (output instanceof Time)
	cell.setContent((Time) output);
      else if (output instanceof Date)
	cell.setContent((Date) output);
      else if (output instanceof String)
	cell.setContent((String) output);
      else
	result = "Don't know how to set cell value for conversion output type:\n" 
	    + Utils.arrayToString(classOut) 
	    + "/" 
	    + ((output != null) ? output.getClass().getName() : "null") + "\n"
	    + "The input that resulted in this output:\n"
	    + Utils.arrayToString(classIn)
	    + "/" 
	    + ((input != null) ? input.getClass().getName() : "null")
	    + "/" + cell.getContent();
    }
    
    return result;
  }
  
  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  protected String doExecute() {
    String			result;
    SpreadSheet			sheetOld;
    SpreadSheet			sheetNew;
    Iterator<CellLocation>	cells;
    
    result = null;

    // is variable attached?
    if (m_GlobalActor == null)
      result = setUpGlobalActor();
    
    sheetOld = (SpreadSheet) m_InputToken.getPayload();
    if (m_NoCopy)
      sheetNew = sheetOld;
    else
      sheetNew = sheetOld.getClone();
    
    cells = m_Finder.findCells(sheetNew);
    if (!m_GlobalActor.getSkip() && !m_GlobalActor.isStopped() && !m_Stopped) {
      synchronized(m_GlobalActor) {
	while (cells.hasNext()) {
	  result = transformCell(cells.next(), sheetNew);
	  if (result != null)
	    break;
	}
      }
    }
    
    if (result == null)
      m_OutputToken = new Token(sheetNew);
    
    return result;
  }
}
