/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SimplePlotGenerator.java
 * Copyright (C) 2013 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.transformer.plotgenerator;

import java.util.ArrayList;
import java.util.List;

import adams.data.spreadsheet.Row;
import adams.data.spreadsheet.SpreadSheet;
import adams.data.spreadsheet.SpreadSheetColumnRange;
import adams.flow.container.SequencePlotterContainer;

/**
 <!-- globalinfo-start -->
 * Plots the numeric data from one or more columns. The plot name is the column name and the X value is the row index.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-default-cell-value &lt;double&gt; (property: defaultCellValue)
 * &nbsp;&nbsp;&nbsp;The default value for missing or non-numeric cells.
 * &nbsp;&nbsp;&nbsp;default: -1.0
 * </pre>
 * 
 * <pre>-columns &lt;java.lang.String&gt; (property: plotColumns)
 * &nbsp;&nbsp;&nbsp;The range of columns to include in the plot; A range is a comma-separated 
 * &nbsp;&nbsp;&nbsp;list of single 1-based indices or sub-ranges of indices ('start-end'); '
 * &nbsp;&nbsp;&nbsp;inv(...)' inverts the range '...'; the following placeholders can be used 
 * &nbsp;&nbsp;&nbsp;as well: first, second, third, last_2, last_1, last
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6866 $
 */
public class SimplePlotGenerator
  extends AbstractPlotGenerator {

  /** for serialization. */
  private static final long serialVersionUID = -1679247321811941656L;
  
  /** the columns to plot. */
  protected SpreadSheetColumnRange m_PlotColumns;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return 
	"Plots the numeric data from one or more columns. "
	+ "The plot name is the column name and the X value is the row index.";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "columns", "plotColumns",
	    "");
  }

  /**
   * Initializes the members.
   */
  @Override
  protected void initialize() {
    super.initialize();

    m_PlotColumns = new SpreadSheetColumnRange();
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("plotColumns");
    if (variable != null)
      return variable;
    else if (m_PlotColumns.getRange().length() > 0)
      return m_PlotColumns.getRange();
    else
      return null;
  }

  /**
   * Sets the column range to use in the plot.
   *
   * @param value	the column range
   */
  public void setPlotColumns(String value) {
    m_PlotColumns.setRange(value);
    reset();
  }

  /**
   * Returns the current column range to use in the plot.
   *
   * @return		the column range
   */
  public String getPlotColumns() {
    return m_PlotColumns.getRange();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String plotColumnsTipText() {
    return "The range of columns to include in the plot; " + m_PlotColumns.getExample();
  }

  /**
   * Checks the spreadsheet.
   * 
   * @param sheet	the sheet to check
   */
  @Override
  protected void check(SpreadSheet sheet) {
    super.check(sheet);

    m_PlotColumns.setSpreadSheet(sheet);
    if (m_PlotColumns.getIntIndices().length == 0)
      throw new IllegalStateException("No plot columns defined/available?");
  }
  
  /**
   * Performs the actual generation of containers.
   * 
   * @param sheet	the basis for the containers
   * @return		the generated containers
   */
  @Override
  protected List<SequencePlotterContainer> doGenerate(SpreadSheet sheet) {
    ArrayList<SequencePlotterContainer>	result;
    SequencePlotterContainer		cont;
    int					i;
    int					n;
    Row					row;
    int[]				plotCols;
    String[]				plotNames;

    result = new ArrayList<SequencePlotterContainer>();

    // create plot containers
    plotCols  = m_PlotColumns.getIntIndices();
    plotNames = new String[plotCols.length];
    for (i = 0; i < plotNames.length; i++)
      plotNames[i] = sheet.getHeaderRow().getCell(plotCols[i]).getContent();
    for (i = 0; i < sheet.getRowCount(); i++) {
      row = sheet.getRow(i);
      for (n = 0; n < plotCols.length; n++) {
	cont = new SequencePlotterContainer(plotNames[n], new Double(i), getCellValue(row, plotCols[n]));
	result.add(cont);
      }
    }

    return result;
  }
}
